﻿/*
 *
 *	Adventure Creator
 *	by Chris Burton, 2013-2021
 *	
 *	"Speech.cs"
 * 
 *	A container class for an active line of dialogue.
 * 
 */

using UnityEngine;
using System.Collections.Generic;

namespace AC
{

	/** A container class for an active line of dialogue. */
	[HelpURL("https://www.adventurecreator.org/scripting-guide/class_a_c_1_1_speech.html")]
	public class Speech
	{

		#region Variables

		/** The line's SpeechLog entry */
		public SpeechLog log;
		/** The display text */
		public string displayText { get; protected set; }
		/** True if the line should play in the backround, and not interrupt Actions or gameplay. */
		public bool isBackground { get; protected set; }
		/** True if the line is active */
		public bool isAlive;
		/** If True, the speech line has an AudioClip supplied */
		public bool hasAudio { get; protected set; }
		/** If True, then the Action that ran this speech will end, but the speech line is still active */
		public bool continueFromSpeech = false;

		/** If True, the assocaited character will not play speaking animation */
		public bool noAnimation = false;
		
		protected int gapIndex = -1;
		protected int continueIndex = -1;
		protected List<SpeechGap> speechGaps = new List<SpeechGap>();
		protected float endTime;
		protected float continueTime;
		protected float minSkipTime;
		protected bool preventSkipping = false;
		protected bool usingRichText = false;

		/** The characters speaking the line, unless a narration */
		public AC.Char speaker { get; protected set; }
		protected bool isSkippable;
		protected bool pauseGap;
		protected bool holdForever = false;

		protected float scrollAmount = 0f;
		protected float pauseEndTime = 0f;
		protected bool pauseIsIndefinite = false;
		protected AudioSource audioSource = null;

		protected bool isRTL = false;

		private List<RichTextTagInstance> richTextTagInstances = new List<RichTextTagInstance>();

		protected int currentCharIndex = 0;
		protected float minDisplayTime;
		protected string realName;

		#endregion


		#region Constructors

		/**
		 * <summary>The default Constructor.</summary>
		 * <param name = "_speaker">The speaking character. If null, the line is considered a narration</param>
		 * <param name = "_message">The subtitle text to display</param>
		 * <param name = "lineID">The unique ID number of the line, as generated by the Speech Manager</param>
		 * <param name = "_isBackground">True if the line should play in the background, and not interrupt Actions or gameplay</param>
		 * <param name = "_noAnimation">True if the speaking character should not play a talking animation</param>
		 * <param name = "_preventSkipping">True if the speech cannot be skipped regardless of subtitle settings in the Speech Manager</param>
		 * <param name = "audioOverride">If set, then this audio will be played instead of the one assigned via the Speech Manager given the line ID and language</param>
		 * <param name = "lipsyncOverride">If set, then this lipsync text asset will be played instead of the one assigned via the Speech Manager given the line ID and language</param>
		 */
		public Speech (Char _speaker, string _message, int lineID, bool _isBackground, bool _noAnimation, bool _preventSkipping = false, AudioClip audioOverride = null, TextAsset lipsyncOverride = null)
		{
			log.Clear ();
			log.lineID = lineID;
			log.fullText = _message;

			isRTL = KickStarter.runtimeLanguages.LanguageReadsRightToLeft (Options.GetLanguageName ());
			isBackground = _isBackground;
			preventSkipping = _preventSkipping;

			realName = string.Empty;
			if (_speaker)
			{
				speaker = _speaker;
				noAnimation = _noAnimation;
				log.speakerName = realName = _speaker.name;

				Player player = _speaker as Player;
				if (player && player.IsActivePlayer () && !KickStarter.speechManager.usePlayerRealName)
				{
					log.speakerName = "Player";
				}

				Hotspot speakerHotspot = _speaker.GetComponent <Hotspot>();
				if (speakerHotspot)
				{
					if (!string.IsNullOrEmpty (speakerHotspot.hotspotName))
					{
						log.speakerName = realName = speakerHotspot.hotspotName;
					}
				}

				if (KickStarter.speechManager.resetExpressionsEachLine)
				{
					_speaker.ClearExpression ();
				}

				if (!_noAnimation)
				{
					if (KickStarter.speechManager.lipSyncMode == LipSyncMode.Off)
					{
						speaker.isLipSyncing = false;
					}
					else if (KickStarter.speechManager.lipSyncMode == LipSyncMode.Salsa2D || KickStarter.speechManager.lipSyncMode == LipSyncMode.FromSpeechText || KickStarter.speechManager.lipSyncMode == LipSyncMode.ReadPamelaFile || KickStarter.speechManager.lipSyncMode == LipSyncMode.ReadSapiFile || KickStarter.speechManager.lipSyncMode == LipSyncMode.ReadPapagayoFile)
					{
						speaker.StartLipSync (KickStarter.dialog.GenerateLipSyncShapes (KickStarter.speechManager.lipSyncMode, lineID, speaker, Options.GetVoiceLanguageName (), _message, lipsyncOverride));
					}
					else if (KickStarter.speechManager.lipSyncMode == LipSyncMode.RogoLipSync)
					{
						AudioSource lipsyncSource = RogoLipSyncIntegration.Play (_speaker, lineID, Options.GetVoiceLanguageName ());

						if (lipsyncSource)
						{
							audioSource = lipsyncSource;
							hasAudio = true;
						}
					}
				}
			}
			else
			{
				speaker = null;			
				log.speakerName = realName = "Narrator";
			}

			if (CanScroll () && KickStarter.speechManager && KickStarter.speechManager.textScrollSpeed <= 0f)
			{
				ACDebug.LogWarning ("Text Scroll Speed must be greater than zero - please amend your Speech Manager");
			}

			// Play sound and time displayDuration to it
			if (audioOverride)
			{
				AssignAudioClip (audioOverride);
			}
			else if (lineID > -1 && !string.IsNullOrEmpty (log.speakerName) && KickStarter.speechManager.searchAudioFiles)
			{
				AudioClip clipObj = KickStarter.runtimeLanguages.GetSpeechAudioClip (lineID, speaker);
				AssignAudioClip (clipObj);
			}

			InitSpeech (_message, true);

			KickStarter.eventManager.Call_OnStartSpeech (this, speaker, log.fullText, log.lineID);
		
			if (CanScroll ())
			{
				KickStarter.eventManager.Call_OnStartSpeechScroll (this, speaker, log.fullText, log.lineID);
			}
		}


		/**
		 * <summary>A special-case Constructor purely used to display text without tags when exporting script-sheets.</summary>
		 * <param name = "_message">The subtitle text to display</param>
		 */
		public Speech (string _message)
		{
			if (Application.isPlaying)
			{
				_message = DetermineGaps (_message);
				_message = DetermineRichTextTags (_message, KickStarter.dialog.richTextTags);
			}

			displayText = _message;
		}


		public Speech (AC.Char _speaker, string _message)
		{
			log.Clear ();

			if (Application.isPlaying)
			{
				_message = DetermineGaps (_message);
				log.textWithRichTextTags = _message;
				_message = DetermineRichTextTags (_message, KickStarter.dialog.richTextTags);
			}

			speaker = _speaker;
			displayText = _message;

			log.fullText = _message;
			log.lineID = -1;

			if (_speaker)
			{
				log.speakerName = realName = _speaker.name;

				Player player = _speaker as Player;
				if (player && player.IsActivePlayer () && !KickStarter.speechManager.usePlayerRealName)
				{
					log.speakerName = "Player";
				}

				Hotspot speakerHotspot = _speaker.GetComponent <Hotspot>();
				if (speakerHotspot)
				{
					if (!string.IsNullOrEmpty (speakerHotspot.hotspotName))
					{
						log.speakerName = realName = speakerHotspot.hotspotName;
					}
				}
			}
			else
			{
				log.speakerName = realName = "Narrator";
			}
		}

		#endregion


		#region PublicFunctions

		/** Updates the speech volume to the level set in Options.  This should be called whenever the Speech volume level is changed. */
		public void UpdateVolume ()
		{
			if (speaker)
			{
				speaker.SetSpeechVolume (Options.optionsData.speechVolume);
			}
		}


		/**
		 * Updates the state of the line.
		 * This is called every LateUpdate call by StateHandler.
		 */
		public void UpdateDisplay ()
		{
			if (minSkipTime > 0f)
			{
				minSkipTime -= Time.deltaTime;
			}

			if (minDisplayTime > 0f)
			{
				minDisplayTime -= Time.deltaTime;
			}

			if (pauseEndTime > 0f)
			{
				pauseEndTime -= Time.deltaTime;
			}

			if (pauseGap)
			{
				if (!pauseIsIndefinite && pauseEndTime <= 0f)
				{
					EndPause ();
				}
				else
				{
					return;
				}
			}
			else
			{
				if (hasAudio && (audioSource == null || !audioSource.isPlaying))
				{
					if (endTime > 0f)
					{
						endTime -= Time.deltaTime;
					}

					if (audioSource == null)
					{
						ACDebug.LogWarning ("No AudioSource found to play speech for " + speaker + " - has their Speech AudioSource been assigned?", speaker);
					}
				}
				else if (!hasAudio)
				{
					if (!CanScroll () || scrollAmount >= 1f)
					{
						if (endTime > 0f)
						{
							endTime -= Time.deltaTime;
						}
					}
				}
			}

			if (hasAudio && !audioSource.isPlaying && Time.timeScale > 0f)
			{
				if (speaker && speaker.isTalking)
				{
					speaker.StopSpeaking ();
					noAnimation = true;
				}
			}

			if (CanScroll ())
			{
				if (scrollAmount < 1f)
				{
					if (!pauseGap)
					{
						scrollAmount += KickStarter.speechManager.textScrollSpeed * Time.deltaTime / 2f / log.fullText.Length;

						if (scrollAmount >= 1f)
						{
							StopScrolling ();
						}

						int newCharIndex = (isRTL)
							? (int) ((1f - scrollAmount) * log.fullText.Length)
							: (int) (scrollAmount * log.fullText.Length);

						if (newCharIndex == 0 || newCharIndex != currentCharIndex)
						{
							currentCharIndex = newCharIndex;
							string newDisplayText = GetTextPortion (log.fullText, currentCharIndex);

							if (!hasAudio && displayText != newDisplayText)
							{
								KickStarter.dialog.PlayScrollAudio (speaker);
							}

							displayText = newDisplayText;
						}
						
						if (gapIndex >= 0 && speechGaps.Count > gapIndex)
						{
							if (HasPassedIndex (speechGaps[gapIndex].characterIndex))
							{
								SetPauseGap ();
								return;
							}
						}

						if (continueIndex >= 0)
						{
							if (HasPassedIndex (continueIndex))
							{
								continueIndex = -1;
								continueFromSpeech = true;
							}
						}
					}
					return;
				}

				if (isRTL)
				{
					displayText = GetTextPortion (log.fullText, 0);
				}
				else
				{
					displayText = GetTextPortion (log.fullText, log.fullText.Length);
				}
			}
			else
			{
				if (gapIndex >= 0 && speechGaps.Count >= gapIndex)
				{
					if (gapIndex == speechGaps.Count)
					{
						displayText = log.fullText;
					}
					else
					{
						float waitTime = (float) speechGaps[gapIndex].waitTime;
						
						if (isRTL)
						{
							displayText = log.fullText.Substring (speechGaps[gapIndex].characterIndex);
						}
						else
						{
							displayText = log.fullText.Substring (0, speechGaps[gapIndex].characterIndex);
						}

						if (waitTime >= 0)
						{
							pauseEndTime = waitTime;
							pauseGap = true;

							speechGaps[gapIndex].CallEvent (this);
						}
						else if (speechGaps[gapIndex].expressionID >= 0)
						{
							speaker.SetExpression (speechGaps [gapIndex].expressionID);
							gapIndex ++;
						}
						else
						{
							pauseIsIndefinite = true;
							pauseGap = true;
						}
					}
				}
				else
				{
					displayText = log.fullText;
				}
				
				if (continueIndex >= 0)
				{
					if (continueTime > 0f)
					{
						continueFromSpeech = true;
					}
				}
			}

			if (endTime <= 0f && minDisplayTime <= 0f)
			{
				if (KickStarter.speechManager.displayForever)
				{
					if (isBackground)
					{
						EndMessage ();
					}
					else
					{
						if (!hasAudio || !audioSource.isPlaying)
						{
							if (!KickStarter.speechManager.playAnimationForever && speaker && speaker.isTalking)
							{
								speaker.StopSpeaking ();
								noAnimation = true;
							}
						}
					}
				}
				else
				{
					if (speaker == null && KickStarter.speechManager.displayNarrationForever)
					{}
					else
					{
						EndMessage ();
					}
				}
			}
		}


		/**
		 * Ends the current pause.
		 */
		public void EndPause ()
		{
			pauseEndTime = 0f;
			pauseGap = false;
			pauseIsIndefinite = false;
			gapIndex ++;
			//scrollAmount = 0f;

			if (CanScroll ())
			{
				KickStarter.eventManager.Call_OnStartSpeechScroll (this, speaker, log.fullText, log.lineID);
			}
		}


		/**
		 * <summary>Checks if the speech line matches certain conditions.</summary>
		 * <param name = "speechMenuLimit">What kind of speech has to play for this Menu to enable (All, BlockingOnly, BackgroundOnly)</param>
		 * <param name = "speechMenuType">What kind of speaker has to be speaking for this Menu to enable (All, CharactersOnly, NarrationOnly, SpecificCharactersOnly)</param>
		 * <param name = "limitToCharacters">A list of character names that this Menu will show for, if speechMenuType = SpeechMenuType.SpecificCharactersOnly</param>
		 * <param name = "speechProximityLimit">Whether or not the distance the Player/MainCamera is from the speaking character affects display</param>
		 * <param name = "speechProximityDistance">The maximum distance if speechProximityLimit != SpeechProximityLimit.NoLimit"</param>
		 * <returns>True if the speech line matches the conditions</returns>
		 */
		public bool HasConditions (SpeechMenuLimit speechMenuLimit, SpeechMenuType speechMenuType, string limitToCharacters, SpeechProximityLimit speechProximityLimit = SpeechProximityLimit.NoLimit, float speechProximityDistance = 0f)
		{
			if (speaker && speechProximityDistance > 0f)
			{
				if (speechProximityLimit == SpeechProximityLimit.LimitByDistanceToCamera)
				{
					float distance = Vector3.Distance (speaker.Transform.position, KickStarter.CameraMainTransform.position);
					if (distance > speechProximityDistance)
					{
						return false;
					}
				}
				else if (speechProximityLimit == SpeechProximityLimit.LimitByDistanceToPlayer && KickStarter.player)
				{
					float distance = Vector3.Distance (speaker.Transform.position, KickStarter.player.transform.position);
					if (distance > speechProximityDistance)
					{
						return false;
					}
				}
			}

			if (!limitToCharacters.StartsWith (";")) limitToCharacters = ";" + limitToCharacters;
			if (!limitToCharacters.EndsWith (";")) limitToCharacters = limitToCharacters + ";";					

			if (speechMenuLimit == SpeechMenuLimit.All ||
			    (speechMenuLimit == SpeechMenuLimit.BlockingOnly && !isBackground) ||
			    (speechMenuLimit == SpeechMenuLimit.BackgroundOnly && isBackground))
			{
				if (speechMenuType == SpeechMenuType.All ||
				    (speechMenuType == SpeechMenuType.CharactersOnly && speaker) ||
				    (speechMenuType == SpeechMenuType.NarrationOnly && speaker == null))
			    {
			    	return true;
			    }
				else if (speechMenuType == SpeechMenuType.SpecificCharactersOnly && speaker)
				{
					if (limitToCharacters.Contains (";" + GetSpeaker (0) + ";"))
					{
						return true;
					}
					else if (limitToCharacters.Contains (";Player;") && speaker && speaker.IsPlayer)
					{
						return true;
					}
				}
				else if (speechMenuType == SpeechMenuType.AllExceptSpecificCharacters && GetSpeakingCharacter ())
				{
					if (limitToCharacters.Contains (";" + GetSpeaker (0) + ";"))
					{
						return false;
					}
					else if (limitToCharacters.Contains (";Player;") && speaker && speaker.IsPlayer)
					{
						return false;
					}
					return true;
				}
			}
			    
			return false;
		}


		/**
		 * <summary>Updates the state of the Speech based on the user's input.
		 * This is called every Update call by StateHandler.</summary>
		 */
		public void UpdateInput ()
		{
			if (isSkippable)
			{
				if (pauseGap && !IsBackgroundSpeech ())
				{
					if (SkipSpeechInput ())
					{
						if (speechGaps[gapIndex].waitTime < 0f)
						{
							KickStarter.playerInput.ResetMouseClick ();
							EndPause ();
						}
						else if (KickStarter.speechManager.allowSpeechSkipping)
						{
							KickStarter.playerInput.ResetMouseClick ();
							EndPause ();
						}
					}
				}
				
				else if ((KickStarter.speechManager.displayForever && !IsBackgroundSpeech ()) ||
						 (KickStarter.speechManager.displayNarrationForever && !IsBackgroundSpeech () && speaker == null))
				{
					if (SkipSpeechInput ())
					{
						KickStarter.playerInput.ResetMouseClick ();
						
						if (KickStarter.stateHandler.gameState == GameState.Cutscene)
						{
							if (KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.DisplayFullText && CanScroll () && displayText != log.textWithRichTextTags)
							{
								// Stop scrolling
								StopScrolling ();

								if (speechGaps != null && speechGaps.Count > gapIndex)
								{
									// Call events
									for (int i=gapIndex; i<speechGaps.Count; i++)
									{
										if (gapIndex >= 0)
										{
											speechGaps[i].CallEvent (this);
										}
									}

									// Find last non-encountered expression
									for (int i=speechGaps.Count-1; i>=gapIndex; i--)
									{
										if (i >= 0 && speechGaps[i].expressionID >= 0)
										{
											speaker.SetExpression (speechGaps[i].expressionID);
											break; // was return
										}
									}
								}

								//
								if (continueIndex >= 0)
								{
									continueIndex = -1;
									continueFromSpeech = true;
								}
								//

								KickStarter.eventManager.Call_OnSkipSpeech (this, true);
							}
							else if (KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.SkipToNextWaitToken && CanScroll () && displayText != log.textWithRichTextTags)
							{
								// Stop scrolling
								if (speechGaps.Count > 0 && speechGaps.Count > gapIndex)
								{
									if (gapIndex < speechGaps.Count && speechGaps[gapIndex].waitTime >= 0)
									{
										speechGaps[gapIndex].CallEvent (this);
									}

									if (gapIndex == speechGaps.Count)
									{
										ExtendTime ();
										StopScrolling ();
									}
									else
									{
										if (isRTL)
										{
											displayText = log.fullText.Substring (speechGaps[gapIndex].characterIndex);
										}
										else
										{
											displayText = log.fullText.Substring (0, speechGaps[gapIndex].characterIndex);
										}
										SetPauseGap ();
									}
								}
								else
								{
									ExtendTime ();
									StopScrolling ();
								}
							}
							else if (KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.DoNothing && CanScroll () && displayText != log.textWithRichTextTags)
							{}
							else
							{
								// Stop message
								EndMessage (true);

								KickStarter.eventManager.Call_OnSkipSpeech (this, false);
							}
						}
						else
						{
							ACDebug.LogWarning ("Cannot skip the line " + log.fullText + " because it is not background speech but the gameState is not a Cutscene! Either mark it as background speech, or initiate a scripted cutscene with AC.KickStarter.stateHandler.StartCutscene ();");
						}
					}
				}
				
				else if (SkipSpeechInput ())
				{
					if ((KickStarter.speechManager.allowSpeechSkipping && !IsBackgroundSpeech ()) ||
						(KickStarter.speechManager.allowSpeechSkipping && KickStarter.speechManager.allowGameplaySpeechSkipping && IsBackgroundSpeech ()) ||
						(KickStarter.speechManager.displayForever && KickStarter.speechManager.allowGameplaySpeechSkipping && IsBackgroundSpeech ()) ||
						(KickStarter.speechManager.displayNarrationForever && speaker == null && KickStarter.speechManager.allowGameplaySpeechSkipping && IsBackgroundSpeech ()))
					{
						KickStarter.playerInput.ResetMouseClick ();
						
						if (KickStarter.stateHandler.gameState == GameState.Cutscene || (KickStarter.speechManager.allowGameplaySpeechSkipping && KickStarter.stateHandler.IsInGameplay ()))
						{
							if ((KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.DisplayFullText || KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.SkipToNextWaitToken) && CanScroll () && displayText != log.textWithRichTextTags)
							{
								// Stop scrolling
								if (speechGaps.Count > 0 && speechGaps.Count > gapIndex)
								{
									while (gapIndex < speechGaps.Count && speechGaps[gapIndex].waitTime >= 0)
									{
										speechGaps[gapIndex].CallEvent (this);

										if (KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.SkipToNextWaitToken)
										{
											break;
										}

										// Find next wait
										gapIndex ++;
									}
									
									if (gapIndex == speechGaps.Count)
									{
										ExtendTime ();
										StopScrolling ();
									}
									else
									{
										if (isRTL)
										{
											displayText = log.fullText.Substring (speechGaps[gapIndex].characterIndex);
										}
										else
										{
											displayText = log.fullText.Substring (0, speechGaps[gapIndex].characterIndex);
										}
										SetPauseGap ();
									}
								}
								else
								{
									ExtendTime ();
									StopScrolling ();
								}
							}
							else if (KickStarter.speechManager.ifSkipWhileScrolling == IfSkipWhileScrolling.DoNothing && CanScroll () && displayText != log.textWithRichTextTags)
							{ }
							else
							{
								EndMessage (true);
							}
						}
					}
				}
			}
		}


		/**
		 * <summary>Ends speech audio, if it is playing in the background.</summary>
		 * <param name = "newSpeaker">If the line's speaker matches this, the audio will not end</param>
		 */
		public void EndBackgroundSpeechAudio (AC.Char newSpeaker)
		{
			if (isBackground && hasAudio && speaker && speaker != newSpeaker)
			{
				if (speaker.speechAudioSource)
				{
					speaker.speechAudioSource.Stop ();
				}
			}
		}


		/** Ends speech audio, regardless of conditions. */
		public void EndSpeechAudio ()
		{
			if (audioSource)
			{
				audioSource.Stop ();
			}
		}


		/**
		 * <summary>Gets the display name of the speaking character.</summary>
		 * <param name = "languageNumber">The index number of the language number to get the text in</param>
		 * <returns>The display name of the speaking character</returns>
		 */
		public string GetSpeaker (int languageNumber = 0)
		{
			if (speaker)
			{
				return speaker.GetName (languageNumber);
			}
			
			return string.Empty;
		}


		/**
		 * <summary>Gets the colour of the subtitle text.</summary>
		 * <returns>The colour of the subtitle text</returns>
		 */
		public Color GetColour ()
		{
			if (speaker)
			{
				return speaker.speechColor;
			}
			return Color.white;
		}
		

		/**
		 * <summary>Gets the speaking character.</summary>
		 * <returns>The speaking character</returns>
		 */
		public AC.Char GetSpeakingCharacter ()
		{
			return speaker;
		}


		/** Checks if the speech line is temporarily paused, due to a [wait] or [wait:X] token. */
		public bool IsPaused ()
		{
			return pauseGap;
		}


		/** Checks if the speech line is currently scrolling */
		public bool IsScrolling ()
		{
			if (CanScroll ())
			{
				return (scrollAmount < 1f);
			}
			return false;
		}


		/** Checks if the speech line is currently playing audio */
		public bool IsPlayingAudio ()
		{
			if (hasAudio && audioSource)
			{
				return audioSource.isPlaying;
			}
			return false;
		}


		/**
		 * <summary>Gets a Sprite based on the portrait graphic of the speaking character.
		 * If lipsincing is enabled, the sprite will be based on the current phoneme.</summary>
		 * <returns>The speaking character's portrait sprite</returns>
		 */
		public Sprite GetPortraitSprite ()
		{
			if (speaker)
			{
				return speaker.GetPortraitSprite ();
			}
			return null;
		}
		

		/**
		 * <summary>Gets the portrait graphic of the speaking character.</summary>
		 * <returns>The speaking character's portrait graphic</returns>
		 */
		public Texture GetPortrait ()
		{
			if (speaker && speaker.GetPortrait ().texture)
			{
				return speaker.GetPortrait ().texture;
			}
			return null;
		}
		

		/**
		 * <summary>Checks if the speaking character's portrait graphic can be animated.</summary>
		 * <returns>True if the character's portrait graphic can be animated</returns>
		 */
		public bool IsAnimating ()
		{
			if (noAnimation)
			{
				return false;
			}

			if (IsPaused () && string.IsNullOrEmpty (displayText))
			{
				return false;
			}
			if (!CanScroll () || KickStarter.speechManager.LipSyncingIsAudioBased ())
			{
				return true;
			}
			return !IsPaused ();
		}


		/**
		 * <summary>Replaces the speech's display text.  Note that this will not affect audio or lipsyncing</summary>
		 * <param name = "newSpeechText">The new display text</param>
		 * <param name = "resetScrollAmount">If True, then the amount by which the text has scrolled will be reset</param>
		 */
		public void ReplaceDisplayText (string newDisplayText, bool resetScrollAmount = true)
		{
			if (!string.IsNullOrEmpty (newDisplayText))
			{
				InitSpeech (newDisplayText, resetScrollAmount);
			}
		}


		/**
		 * <summary>Checks if a Menu is able to show this speech line.</summary>
		 * <param name = "menu">The Menu to check against</param>
		 * <returns>True if the Menu is able to show this speech line</returns>
		 */
		public bool MenuCanShow (Menu menu)
		{
			if (menu != null)
			{
				return HasConditions (menu.speechMenuLimit, menu.speechMenuType, menu.limitToCharacters, menu.speechProximityLimit, menu.speechProximityDistance);
			}
			return false;
		}


		/** Checks if scrolling is possible with this line - regardless of whether or not it is currently doing so */
		public bool CanScroll ()
		{
			if (speaker == null)
			{
				return KickStarter.speechManager.scrollNarration;
			}
			return KickStarter.speechManager.scrollSubtitles;
		}

		#endregion


		#region ProtectedFunctions

		protected void ExtendTime ()
		{
			if (CanScroll () && !KickStarter.speechManager.scrollingTextFactorsLength && !hasAudio)
			{
				// Extend length of speech display, since its been skipped prematurely
				float timeExtension = (1f - scrollAmount) * KickStarter.speechManager.screenTimeFactor * GetLengthWithoutRichText (log.fullText);
				endTime = Mathf.Max (endTime, timeExtension);
			}
		}


		protected void StopScrolling ()
		{
			scrollAmount = 1f;
			displayText = log.textWithRichTextTags;
			
			if (holdForever)
			{
				continueFromSpeech = true;
			}

			// Call events
			KickStarter.eventManager.Call_OnEndSpeechScroll (this, speaker, log.fullText, log.lineID);
			KickStarter.eventManager.Call_OnCompleteSpeechScroll (this, speaker, log.fullText, log.lineID);
		}


		protected void SetPauseGap ()
		{
			scrollAmount = (float) speechGaps [gapIndex].characterIndex / (float) log.fullText.Length;
			if (isRTL)
			{
				scrollAmount = 1f - scrollAmount;
			}

			float waitTime = speechGaps [gapIndex].waitTime;
			pauseGap = true;
			pauseIsIndefinite = false;

			if (speechGaps [gapIndex].pauseIsIndefinite)
			{
				pauseEndTime = 0f;
				pauseIsIndefinite = true;
			}
			else if (waitTime >= 0f)
			{
				pauseEndTime = waitTime;
				speechGaps[gapIndex].CallEvent (this);
			}
			else if (speechGaps [gapIndex].expressionID >= 0)
			{
				pauseEndTime = 0f;
				speaker.SetExpression (speechGaps [gapIndex].expressionID);
			}
			else
			{
				pauseEndTime = 0f;
			}

			if (pauseEndTime > 0f || pauseIsIndefinite)
			{
				// Call event
				KickStarter.eventManager.Call_OnEndSpeechScroll (this, speaker, log.fullText, log.lineID);
			}
		}


		protected string DetermineGaps (string _text)
		{
			speechGaps.Clear ();
			continueIndex = -1;

			if (!string.IsNullOrEmpty (_text))
			{
				string[] eventKeys = KickStarter.dialog.SpeechEventTokenKeys;

				for (int i=0; i<_text.Length; i++)
				{
					string textPortion1 = _text.Substring (0, i);
					string textPortion2 = _text.Substring (i);

					if (textPortion2.StartsWith ("[continue]"))
					{
						continueIndex = i;

						_text = textPortion1 + textPortion2.Substring ("[continue]".Length);
						CorrectPreviousGaps (continueIndex, 10);

						i = -1;
						continue;
					}
					else if (textPortion2.StartsWith ("[hold]"))
					{
						if (continueIndex == -1)
						{
							continueIndex = i;
						}

						_text = textPortion1 + textPortion2.Substring ("[hold]".Length);
						holdForever = true;
						CorrectPreviousGaps (continueIndex, 6);

						i = -1;
						continue;
					}
					else if (textPortion2.StartsWith ("[expression:") && speaker)
					{
						// Expression change
						int endIndex = textPortion2.IndexOf ("]");
						string expressionText = textPortion2.Substring (12, endIndex - 12);
						int expressionID = speaker.GetExpressionID (expressionText);

						speechGaps.Add (new SpeechGap (i, expressionID));
						_text = textPortion1 + textPortion2.Substring (endIndex + 1);

						i = -1;
						continue;
					}
					else if (textPortion2.StartsWith ("[wait]"))
					{
						// Indefinite wait
						speechGaps.Add (new SpeechGap (i, true));
						_text = textPortion1 + textPortion2.Substring ("[wait]".Length);

						i = -1;
						continue;
					}
					else if (textPortion2.StartsWith ("[wait:"))
					{
						// Timed wait
						int endIndex = textPortion2.IndexOf ("]");
						string waitTimeText = textPortion2.Substring (6, endIndex - 6);

						speechGaps.Add (new SpeechGap (i, FloatParse (waitTimeText)));
						_text = textPortion1 + textPortion2.Substring (endIndex + 1);

						i = -1;
						continue;
					}
					else if (eventKeys != null)
					{
						foreach (string eventKey in eventKeys)
						{
							if (string.IsNullOrEmpty (eventKey)) continue;

							string keyStart = "[" + eventKey + ":";
							if (textPortion2.StartsWith (keyStart))
							{
								int endIndex = textPortion2.IndexOf ("]");
								string eventValue = textPortion2.Substring (keyStart.Length, endIndex - keyStart.Length);

								speechGaps.Add (new SpeechGap (i, eventKey, eventValue));
								string replacementText = KickStarter.eventManager.Call_OnRequestSpeechTokenReplacement (this, eventKey, eventValue);
								_text = textPortion1 + replacementText + textPortion2.Substring (endIndex + 1);

								i = -1;
								continue;
							}
						}
					}
				}
			}

			// Sort speechGaps
			if (speechGaps.Count > 1)
			{
				if (isRTL)
				{
					speechGaps.Sort (delegate (SpeechGap b, SpeechGap a) {return a.characterIndex.CompareTo (b.characterIndex);});
				}
				else
				{
					speechGaps.Sort (delegate (SpeechGap a, SpeechGap b) { return a.characterIndex.CompareTo (b.characterIndex); });
				}
			}
			
			return _text;
		}


		protected string DetermineRichTextTags (string _text, string[] tagNames)
		{
			if (!CanScroll ())
			{
				return _text;
			}

			List<RichTextTag> richTextTags = new List<RichTextTag>();
			foreach (string tagName in tagNames)
			{
				if (!string.IsNullOrEmpty (tagName))
				{
					RichTextTag newTag = new RichTextTag (tagName);
					if (!richTextTags.Contains (newTag))
					{
						richTextTags.Add (newTag);
					}
				}
			}

			richTextTagInstances = new List<RichTextTagInstance>();

			if (!string.IsNullOrEmpty (_text))
			{
				for (int i=0; i<_text.Length; i++)
				{
					string textPortion1 = _text.Substring (0, i);
					string textPortion2 = _text.Substring (i);

					foreach (RichTextTag richTextTag in richTextTags)
					{
						if (textPortion2.StartsWith (richTextTag.openTag))
						{
							int indexToClose = textPortion2.IndexOf (">") + 1;
							if (indexToClose > 1)
							{
								string openTag = textPortion2.Substring (0, indexToClose);
								_text = textPortion1 + textPortion2.Substring (openTag.Length);
								CorrectPreviousGaps (i, openTag.Length);

								richTextTagInstances.Add (new RichTextTagInstance (openTag, richTextTag.closeTag, i));
								i = -1;
								usingRichText = true;
								continue;
							}
						}
						else if (textPortion2.StartsWith (richTextTag.closeTag))
						{
							_text = textPortion1 + textPortion2.Substring (richTextTag.closeTag.Length);
							CorrectPreviousGaps (i, richTextTag.closeTag.Length);

							// Now go backward through invalid instances to update it
							for (int j=richTextTagInstances.Count-1; j>=0; j--)
							{
								if (!richTextTagInstances[j].IsValid () &&
									richTextTagInstances[j].closeText == richTextTag.closeTag)
								{
									richTextTagInstances[j] = new RichTextTagInstance (richTextTagInstances[j], i);
								}
							}

							i = -1;
							continue;
						}
					}
				}
			}

			return _text;
		}


		protected string FindSpeakerTag (string _message, string _speakerName)
		{
			if (!string.IsNullOrEmpty (_message))
			{
				if (_message.Contains ("[speaker]"))
				{
					_message = _message.Replace ("[speaker]", _speakerName);
				}
			}
			return _message;
		}


		protected void CorrectPreviousGaps (int minCharIndex, int offset)
		{
			if (speechGaps.Count > 0)
			{
				for (int i=0; i<speechGaps.Count; i++)
				{
					if (speechGaps[i].characterIndex > minCharIndex)
					{
						SpeechGap speechGap = speechGaps[i];
						speechGap.characterIndex -= offset;
						speechGaps[i] = speechGap;
					}
				}
			}
		}


		protected float FloatParse (string text)
		{
			float _value = 0f;
			if (!string.IsNullOrEmpty (text))
			{
				float.TryParse (text, out _value);
			}
			return _value;
		}


		protected bool IsBackgroundSpeech ()
		{
			return isBackground;
		}


		protected void EndMessage (bool forceOff = false)
		{
			if (holdForever)
			{
				continueFromSpeech = true;
				return;
			}
			endTime = 0f;
			isSkippable = false;

			if (speaker)
			{
				speaker.StopSpeaking ();
			}

			EndSpeechAudio ();

			if (!forceOff && gapIndex >= 0 && gapIndex < speechGaps.Count)
			{
				gapIndex ++;
			}
			else
			{
				isAlive = false;
				KickStarter.stateHandler.UpdateAllMaxVolumes ();
			}
		}


		protected bool SkipSpeechInput ()
		{
			if (minSkipTime > 0f || preventSkipping)
			{
				return false;
			}

			if (holdForever && log.textWithRichTextTags == displayText)
			{
				return false;
			}

			if (KickStarter.speechManager.canSkipWithMouseClicks && (KickStarter.playerInput.GetMouseState () == MouseState.SingleClick ||
			    													 KickStarter.playerInput.GetMouseState () == MouseState.RightClick))
			{
				return true;
			}

			if (KickStarter.playerInput.InputGetButtonDown ("SkipSpeech"))
			{
				return true;
			}

			return false;
		}


		protected void InitSpeech (string _message, bool resetScrollAmount)
		{
			gapIndex = -1;
			continueIndex = -1;
			speechGaps = new List<SpeechGap>();
			endTime = 0f;
			continueTime = 0f;
			minSkipTime = 0f;
			usingRichText = false;

			isSkippable = false;
			pauseGap = false;
			holdForever = false;

			if (resetScrollAmount)
			{
				scrollAmount = 0f;
			}
			pauseEndTime = 0f;
			pauseIsIndefinite = false;
		
			richTextTagInstances = new List<RichTextTagInstance>();

			currentCharIndex = 0;
			minDisplayTime = 0f;

			if (Application.isPlaying)
			{
				_message = FindSpeakerTag (_message, realName);
				_message = DetermineGaps (_message);
				log.textWithRichTextTags = _message;
				_message = DetermineRichTextTags (_message, KickStarter.dialog.richTextTags);
			}

			gapIndex = (speechGaps.Count > 0) ? 0 : -1;

			float displayDuration = 0f;
			if (hasAudio)
			{
				displayDuration = KickStarter.speechManager.screenTimeFactor * 5f;
			}
			else if (!CanScroll () || KickStarter.speechManager.scrollingTextFactorsLength)
			{
				float totalWaitTime = GetLengthWithoutRichText (_message) - GetTotalWaitTokenDuration ();
				if (totalWaitTime < 0f)
				{
					totalWaitTime = 0.1f;
				}

				displayDuration = KickStarter.speechManager.screenTimeFactor * totalWaitTime;
			}
			else
			{
				displayDuration = KickStarter.speechManager.screenTimeFactor * 5f;
			}

			displayDuration = Mathf.Max (displayDuration, 0.1f);
			log.fullText = _message;

			if (!CanScroll ())
			{
				if (continueIndex > 0)
				{
					continueTime = (continueIndex / KickStarter.speechManager.textScrollSpeed);
				}
				
				if (speechGaps.Count > 0)
				{
					displayText = log.fullText.Substring (0, speechGaps[0].characterIndex);
				}
				else
				{
					displayText = log.fullText;
				}
			}
			else
			{
				displayText = " ";
			}

			isAlive = true;
			isSkippable = true;
			pauseGap = false;
			endTime = displayDuration;

			minSkipTime = KickStarter.speechManager.skipThresholdTime;
			minDisplayTime = Mathf.Max (0f, KickStarter.speechManager.minimumDisplayTime);

			if (hasAudio && KickStarter.speechManager.syncSubtitlesToAudio)
			{
				if (KickStarter.speechManager.displayForever || (speaker == null && KickStarter.speechManager.displayNarrationForever))
				{}
				else
				{
					minDisplayTime = 0f;
					endTime = 0.1f;
				}
			}

			if (endTime <= 0f)
			{
				EndMessage ();
			}
		}


		protected bool HasPassedIndex (int indexToCheck)
		{
			if (isRTL)
			{
				return (currentCharIndex <= indexToCheck);
			}
			return (currentCharIndex >= indexToCheck);
		}


		protected string GetTextPortion (string fullText, int index)
		{
			if (index <= 0)
			{
				if (isRTL)
				{
					index = 0;
				}
				else
				{
					return string.Empty;
				}
			}

			if (index > fullText.Length)
			{
				index = fullText.Length;
			}

			if (!usingRichText)
			{
				// No rich tags right now, so don't do anything complicated
				if (isRTL)
				{
					return fullText.Substring (index);
				}
				return fullText.Substring (0, index);
			}

			if (isRTL)
			{
				string newText = fullText.Substring (index);
				int length = newText.Length;
				int inverse = fullText.Length - length;

				string prefix = string.Empty;

				for (int i=fullText.Length; i>=inverse; i--)
				{
					int closingTagOffset = 0;

					for (int j=richTextTagInstances.Count-1; j>=0; j--)
					{
						RichTextTagInstance instance = richTextTagInstances[j];

						if (instance.IsValid ())
						{
							if (instance.startIndex == i)
							{
								newText = newText.Insert (i - index, instance.openText);
								continue;
							}
							else if (instance.endIndex == i)
							{
								newText = newText.Insert (i - index + closingTagOffset, instance.closeText);

								if (instance.startIndex < inverse)
								{
									prefix += instance.openText;
								}
								else
								{
									closingTagOffset += instance.closeText.Length;
								}
								continue;
							}
						}
					}
				}

				return prefix + newText;
			}
			else
			{
				string newText = fullText.Substring (0, index);

				for (int i=index; i>=0; i--)
				{
					int closingTagOffset = 0;

					for (int j=richTextTagInstances.Count-1; j>=0; j--)
					{
						RichTextTagInstance instance = richTextTagInstances[j];

						if (instance.IsValid ())
						{
							if (instance.endIndex == i)
							{
								newText = newText.Insert (i + closingTagOffset, instance.closeText);
								closingTagOffset += instance.closeText.Length;
								continue;
							}
							else if (instance.startIndex == i)
							{
								if (instance.endIndex > index)
								{
									newText += instance.closeText;
								}

								newText = newText.Insert (i, instance.openText);
								continue;
							}
						}
					}
				}
				return newText;
			}
		}


		protected float GetLengthWithoutRichText (string _message)
		{
			_message = _message.Replace ("[var:", string.Empty);
			_message = _message.Replace ("[localvar:", string.Empty);
			_message = _message.Replace ("[wait:", string.Empty);
			_message = _message.Replace ("[continue:", string.Empty);
			_message = _message.Replace ("[expression:", string.Empty);
			_message = _message.Replace ("[paramlabel:", string.Empty);
			_message = _message.Replace ("[param:", string.Empty);

			return (float) _message.Length;
		}


		protected float GetTotalWaitTokenDuration ()
		{
			if (speechGaps != null && speechGaps.Count > 0f)
			{
				float totalDuration = 0f;
				foreach (SpeechGap speechGap in speechGaps)
				{
					if (speechGap.waitTime > 0f)
					{
						totalDuration += speechGap.waitTime;
					}
				}

				return totalDuration;
			}

			return 0f;
		}


		protected void AssignAudioClip (AudioClip audioClip)
		{
			if (audioClip)
			{
				audioSource = null;

				if (speaker)
				{
					if (!noAnimation && KickStarter.speechManager.lipSyncMode == LipSyncMode.FaceFX)
					{
						FaceFXIntegration.Play (speaker, log.speakerName + log.lineID, audioClip);
					}

					if (speaker.speechAudioSource)
					{
						audioSource = speaker.speechAudioSource;
						speaker.SetSpeechVolume (Options.optionsData.speechVolume);
					}
					else
					{
						ACDebug.LogWarning (speaker.name + " has no audio source component!", speaker);
					}
				}
				else
				{
					audioSource = KickStarter.dialog.GetNarratorAudioSource ();

					if (audioSource == null)
					{
						ACDebug.LogWarning ("Cannot play audio for speech line '" + log.fullText + "' as there is no AudioSource - assign a new 'Default Sound' in the Scene Manager.");
					}
				}

				if (audioSource)
				{
					audioSource.clip = audioClip;
					audioSource.loop = false;
					audioSource.Play ();
					hasAudio = true;
				}
			}
		}

		#endregion


		#region GetSet

		/** The display name of the speaking character */
		public string SpeakerName
		{
			get
			{
				return log.speakerName;
			}
		}


		/** The ID number of the line, as set by the Speech Manager */
		public int LineID
		{
			get
			{
				return log.lineID;
			}
		}


		/** The full display text of the line */
		public string FullText
		{
			get
			{
				return log.fullText;
			}
		}


		/** The line's associated SpeechLine class, provided it's been gathered by the Speech Manager */
		public SpeechLine SpeechLine
		{
			get
			{
				return KickStarter.speechManager.GetLine (LineID);
			}
		}

		#endregion


		#region PrivateStructs

		private struct RichTextTag
		{

			public string openTag;
			public string closeTag;
			public bool usesParameters;


			public RichTextTag (string tag)
			{
				if (tag.Contains ("="))
				{
					usesParameters = true;
					openTag = "<" + tag;
					closeTag = "</" + tag.Substring (0, tag.Length-1) + ">";
				}
				else
				{
					usesParameters = false;
					openTag = "<" + tag + ">";
					closeTag = "</" + tag + ">";
				}
			}
		}


		private struct RichTextTagInstance
		{

			public int startIndex;
			public int endIndex;
			public string openText;
			public string closeText;


			public RichTextTagInstance (string _openText, string _closeText, int _startIndex)
			{
				startIndex = _startIndex;
				endIndex = 0;
				openText = _openText;
				closeText = _closeText;
			}


			public RichTextTagInstance (RichTextTagInstance instance, int _endIndex)
			{
				startIndex = instance.startIndex;
				openText = instance.openText;
				closeText = instance.closeText;
				endIndex = _endIndex;
			}


			public bool IsValid ()
			{
				if (!string.IsNullOrEmpty (openText) &&
					!string.IsNullOrEmpty (closeText) &&
					endIndex > startIndex)
				{
					return true;
				}
				return false;
			}

		}

		#endregion

	}


	/**
	 * A data struct for an entry in the game's speech log.
	 */
	public struct SpeechLog
	{

		/** The full display text of the line */
		public string fullText;
		/** The display name of the speaking character */
		public string speakerName;
		/** The ID number of the line, as set by the Speech Manager */
		public int lineID;
		/** The original text, with rich text tags intact */
		public string textWithRichTextTags;


		/**
		 * Clears the struct.
		 */
		public void Clear ()
		{
			fullText = string.Empty;
			speakerName = string.Empty;
			lineID = -1;
		}

	}


	/**
	 * A data container for an label a 'Dialogue: Play speech' Action can be tagged as
	 */
	[System.Serializable]
	public class SpeechTag
	{
		
		/** A unique identified */
		public int ID;
		/** The tag's text */
		public string label;
		
		
		/**
		 * <summary>The default Constructor.</summary>
		 * <param name = "idArray">An array of already-used ID numbers, so that a unique one can be generated</param>
		 */
		public SpeechTag (int[] idArray)
		{
			ID = 0;
			label = string.Empty;
			
			// Update id based on array
			if (idArray != null && idArray.Length > 0)
			{
				foreach (int _id in idArray)
				{
					if (ID == _id)
						ID ++;
				}
			}
		}


		/**
		 * <summary>A Constructor for the first SpeechTag.</summary>
		 * <param name = "_label">The SpeechTag's label</param>
		 */
		public SpeechTag (string _label)
		{
			ID = 0;
			label = _label;
		}
		
	}

}