/*
 *
 *	Adventure Creator
 *	by Chris Burton, 2013-2021
 *	
 *	"Hotspot.cs"
 * 
 *	This script handles all the possible
 *	interactions on both hotspots and NPCs.
 * 
 */

using UnityEngine;
using System.Collections;
using System.Collections.Generic;

namespace AC
{

	/**
	 * This component provides the player with a region of space in the scene that can be interacted with.
	 * Data for each interaction is stored within the Button class, and this component stores them in Lists.
	 * The number of interactions, and how exactly they are triggered, are determined in SettingsManager.
	 */
	[AddComponentMenu("Adventure Creator/Hotspots/Hotspot")]
	[HelpURL("https://www.adventurecreator.org/scripting-guide/class_a_c_1_1_hotspot.html")]
	public class Hotspot : MonoBehaviour, ITranslatable
	{

		#region Variables

		/** The source of the commands that are run when an option is chosen (InScene, AssetFile, CustomScript) */	
		public AC.InteractionSource interactionSource;
		/** If assigned, then the Hotspot will only be interactive when the assigned _Camera is active */
		public _Camera limitToCamera = null;
		/** If assigned, then the Hotspot will only be interactive when the player is within this Trigger Collider's boundary */
		public InteractiveBoundary interactiveBoundary = null;

		/** The display name, if not the GameObject's name */
		public string hotspotName;
		/** The translation ID number of the Hotspot's name, as generated by SpeechManager */
		public int lineID = -1;
		/** The Highlight component that controls any highlighting effects associated with the Hotspot */
		public Highlight highlight;
		/** The Marker that the player can optionally automatically walk to before an Interaction runs */
		public Marker walkToMarker;
		/** A Transform that represents the centre of the Hotspot, if it is not physically at the same point as the Hotspot's GameObject itself */
		public Transform centrePoint;
		/** What the centrePoint overrides, if set */
		public CentrePointOverrides centrePointOverrides = CentrePointOverrides.FacingAndIconPosition;

		/** If True, then the Hotspot can have 'Use" interactions */
		public bool provideUseInteraction;
		/** No longer used by Adventure Creator, but kept so that older projects can be upgraded */
		public Button useButton = new Button();

		/** A List of all available 'Use' interactions */
		public List<Button> useButtons = new List<Button>();
		/** If True, then clicking the Hotspot will run the Hotspot's first interaction in useButtons, regardless of the interactionMethod chosen in SettingsManager */
		public bool oneClick = false;

		/** If True, then the Hotspot can have an 'Examine' interaction, if interactionMethod = AC_InteractionMethod.ContextSensitive in SettingsManager */
		public bool provideLookInteraction;
		/** The 'Examine' interaction, if interactionMethod = AC_InteractionMethod.ContextSensitive in SettingsManager */
		public Button lookButton = new Button();

		/** If True, then the Hotspot can have 'Inventory' interactions */
		public bool provideInvInteraction;
		/** A List of all available 'Inventory' interactions */
		public List<Button> invButtons = new List<Button>();

		/** If True, then the Hotspot can have an unhandled 'Use' interaction */
		public bool provideUnhandledUseInteraction;
		/** If True, then the Hotspot can have an unhandled 'Inventory' interaction */
		public bool provideUnhandledInvInteraction;
		/** The unhandled 'Use' interaction, which will be run if the player interacts with the Hotspot using a cursor that has not been defined in the Hotspot's 'Use interactions' panel. */
		public Button unhandledUseButton = new Button();
		/** The unhandled 'Inventory' interaction, which will be run if the player uses an inventory item on the Hotspot, and it is not handled within invButtons */
		public Button unhandledInvButton = new Button();

		/** If True, the Hotspot's label will be lower-cased when placed in the middle of a Hotspot label if it is not at the start. */
		public bool canBeLowerCase = false;

		/** If True, then a Gizmo may be drawn in the Scene window at the Hotspots's position */
		public bool drawGizmos = true;

		/** The index of the last-active interaction */
		public int lastInteractionIndex = 0;
		/** The translation ID number of the Hotspot's name, if it was changed mid-game */
		public int displayLineID = -1;
		/** The 'Sorting Layer' of the icon's SpriteRenderer, if drawn in World Space */
		public string iconSortingLayer = "";
		/** The 'Order in Layer' of the icon's SpriteRenderer, if drawn in World Space */
		public int iconSortingOrder = 0;

		/** The effect that double-clicking on the Hotspot has, if interactionMethod = AC_InteractionMethod.ContextSensitive in SettingsManager */
		public DoubleClickingHotspot doubleClickingHotspot = DoubleClickingHotspot.MakesPlayerRun;

		/** If True, then the player will turn their head when the Hotspot is selected (if SettingsManager's playerFacesHotspots = True) */
		public bool playerTurnsHead = true;

		protected Collider _collider;
		protected Collider2D _collider2D;
		protected bool isOn = true;
		protected float iconAlpha = 0;
		protected Sprite iconSprite = null;
		protected SpriteRenderer iconRenderer = null;
		protected CursorIcon mainIcon;

		protected LerpUtils.FloatLerp iconAlphaLerp = new LerpUtils.FloatLerp (true);

		protected float manualShowIconSpeed = 5f;
		protected bool manuallyShowIcon = false;
		protected bool tooFarAway = false;

		protected MatchingInvInteractionData matchingInvInteractionData;
		private Transform _transform;

		#endregion


		#region UnityStandards

		protected void Awake ()
		{
			if (KickStarter.settingsManager && KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.ContextSensitive)
			{
				UpgradeSelf ();
			}
			
			_collider = GetComponent <Collider>();
			_collider2D = GetComponent <Collider2D>();

			lastInteractionIndex = FindFirstEnabledInteraction ();
			displayLineID = lineID;
		}


		protected void OnEnable ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Register (this);

			EventManager.OnSwitchCamera += OnSwitchCamera;
		}


		protected void Start ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Register (this);
		}


		protected void OnDisable ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Unregister (this);

			EventManager.OnSwitchCamera -= OnSwitchCamera;
		}

		#endregion


		#region PublicFunctions

		/** Registers this as the active Hotspot in the PlayerInteraction script */
		public void SetAsActive ()
		{
			KickStarter.playerInteraction.SetActiveHotspot (this);
		}


		/** Unregisters this as the active Hotspot in the PlayerInteraction script */
		public void SetAsInactive ()
		{ 
			if (KickStarter.playerInteraction.GetActiveHotspot () == this)
			{
				KickStarter.playerInteraction.SetActiveHotspot (null);
			}
		}


		/**
		 * <summary>Runs the Hotspot's 'Examine' interaction, if one is defined.</summary>
		 */
		public void RunExamineInteraction ()
		{
			if (lookButton != null)
			{
				KickStarter.playerInteraction.ExamineHotspot (this);
			}
		}


		/**
		 * <summary>Runs one of the Hotspot's 'Use' interactions.</summary>
		 * <param name = "iconID">The ID number of the CursorIcon associated with the Button. If no number is supplied, the first enabled Use interaction will be run.</param>
		 */
		public void RunUseInteraction (int iconID = -1)
		{
			if (useButtons == null || useButtons.Count == 0)
			{
				return;
			}
			
			iconID = Mathf.Max (-1, iconID);
			KickStarter.playerInteraction.UseHotspot (this, iconID);
		}


		/**
		 * <summary>Runs one of the Hotspot's 'Inventory' interactions.</summary>
		 * <param name = "invID">The ID number of the InvItem associated with the Button. If no number is supplied, that of the currently-selected inventory item will be used.</param>
		 */
		public void RunInventoryInteraction (int invID = -1)
		{
			if (invID < 0)
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, KickStarter.runtimeInventory.SelectedInstance, false);
			}
			else
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, new InvInstance (invID), false);
			}
		}


		/**
		 * <summary>Runs one of the Hotspot's 'Inventory' interactions.</summary>
		 * <param name = "invItem">The inventory item associated with the Button. If no item is supplied, that of the currently-selected inventory item will be used.</param>
		 */
		public void RunInventoryInteraction (InvItem invItem = null)
		{
			if (invItem == null)
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, KickStarter.runtimeInventory.SelectedInstance, false);
			}
			else
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, new InvInstance (invItem), false);
			}
		}


		/**
		 * <summary>Runs one of the Hotspot's 'Inventory' interactions.</summary>
		 * <param name = "invInstance">The inventory instance associated with the Button. If no item is supplied, that of the currently-selected inventory item will be used.</param>
		 */
		public void RunInventoryInteraction (InvInstance invInstance = null)
		{
			if (invInstance == null)
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, KickStarter.runtimeInventory.SelectedInstance, false);
			}
			else
			{
				KickStarter.playerInteraction.UseInventoryOnHotspot (this, invInstance, false);
			}
		}


		/**
		 * <summary>Highlights the Hotspot based on the mouse cursor's proximity.</summary>
		 * <param name = "isGameplay">If True, then it is during gameplay, and the highlight effect will work</param>
		 */
		public void SetProximity (bool isGameplay)
		{
			if (highlight)
			{
				if (gameObject.layer == LayerMask.NameToLayer (KickStarter.settingsManager.deactivatedLayer) ||
					!isGameplay ||
					!IsOn ())
				{
					highlight.SetMinHighlight (0f);
				}
				else
				{
					float amount = (GetIconScreenPosition () - KickStarter.playerInput.GetMousePosition ()).magnitude / ACScreen.safeArea.size.magnitude;
					highlight.SetMinHighlight (1f - (amount * KickStarter.settingsManager.highlightProximityFactor));
				}
			}
		}
		

		/**
		 * <summary>Upgrades the Hotspot from a previous version of Adventure Creator.</summary>
		 * <returns>True if the upgrade was successful</returns>
		 */
		public bool UpgradeSelf ()
		{
			if (useButton.IsButtonModified ())
			{
				Button newUseButton = new Button ();
				newUseButton.CopyButton (useButton);
				useButtons.Add (newUseButton);
				useButton = new Button ();
				provideUseInteraction = true;

				if (Application.isPlaying)
				{
					ACDebug.Log ("Hotspot '" + gameObject.name + "' has been temporarily upgraded - please view its Inspector when the game ends and save the scene.", gameObject);
				}
				else
				{
					ACDebug.Log ("Upgraded Hotspot '" + gameObject.name + "', please save the scene.", gameObject);
				}

				return true;
			}
			return false;
		}
		

		/**
		 * <summary>Draws an icon at the Hotspot's centre.</summary>
		 * <param name = "inWorldSpace">If True, the icon shall be drawn as a sprite in world space, as opposed to an OnGUI graphic in screen space.</param>
		 */
		public void DrawHotspotIcon (bool inWorldSpace = false)
		{
			if (iconAlpha > 0f)
			{
				if (!KickStarter.mainCamera.IsPointInCamera (GetIconScreenPosition ()))
				{
					return;
				}

				if (inWorldSpace)
				{
					if (iconRenderer == null)
					{
						GameObject iconOb = new GameObject (this.name + " - icon");
						iconRenderer = iconOb.AddComponent <SpriteRenderer>();
						iconOb.transform.localScale = Vector3.one * (25f * KickStarter.settingsManager.hotspotIconSize);

						if (iconSortingLayer != "")
						{
							iconRenderer.GetComponent <SpriteRenderer>().sortingLayerName = iconSortingLayer;
						}
						iconRenderer.GetComponent <SpriteRenderer>().sortingOrder = iconSortingOrder;
					}

					if (KickStarter.settingsManager.hotspotIcon == HotspotIcon.UseIcon)
					{
						GetMainIcon ();
						if (mainIcon != null)
						{
							iconRenderer.sprite = mainIcon.GetSprite ();
						}
					}
					else
					{
						if (iconSprite == null && KickStarter.settingsManager.hotspotIconTexture)
						{
							iconSprite = UnityEngine.Sprite.Create (KickStarter.settingsManager.hotspotIconTexture, new Rect (0f, 0f, KickStarter.settingsManager.hotspotIconTexture.width, KickStarter.settingsManager.hotspotIconTexture.height), new Vector2 (0.5f, 0.5f));
						}
						if (iconSprite != iconRenderer.sprite)
						{
							iconRenderer.sprite = iconSprite;
						}
					}

					iconRenderer.transform.position = GetIconPosition ();
					iconRenderer.transform.LookAt (iconRenderer.transform.position + KickStarter.mainCamera.Transform.rotation * Vector3.forward, KickStarter.mainCamera.Transform.rotation * Vector3.up);
				}
				else
				{
					if (iconRenderer)
					{
						Destroy (iconRenderer.gameObject);
						iconRenderer = null;
					}

					Color c = GUI.color;
					Color tempColor = c;
					c.a = iconAlpha;
					GUI.color = c;
					
					if (KickStarter.settingsManager.hotspotIcon == HotspotIcon.UseIcon)
					{
						GetMainIcon ();
						if (mainIcon != null)
						{
							mainIcon.Draw (GetIconScreenPosition (), !KickStarter.playerMenus.IsMouseOverInteractionMenu ());
						}
					}
					else if (KickStarter.settingsManager.hotspotIconTexture)
					{
						GUI.DrawTexture (AdvGame.GUIBox (GetIconScreenPosition (), KickStarter.settingsManager.hotspotIconSize), KickStarter.settingsManager.hotspotIconTexture, ScaleMode.ScaleToFit, true, 0f);
					}
					
					GUI.color = tempColor;
				}
			}

			if (inWorldSpace && iconRenderer != null)
			{
				Color tempColor = iconRenderer.color;
				tempColor.a = iconAlpha;
				iconRenderer.color = tempColor;
			}
		}


		/**
		 * <summary>Gets the label to display when the cursor is over this Hotspot, with cursor names and active inventory item included if appropriate.</summary>
		 * <param name = "languageNumber">The index number of the language to return the line in, where 0 = the game's original language.</param>
		 * <param name = "cursorID">The ID number of the cursor icon (set in the Cursor Manager) to get the label for. If <0, the active cursor will be used.</param>
		 * <returns>The label to display when the cursor is over this Hotspot, with cursor names and active inventory item included if appropriate.</returns>
		 */
		public string GetFullLabel (int languageNumber = 0, int cursorID = -1)
		{
			if (KickStarter.stateHandler.gameState == GameState.DialogOptions && !KickStarter.settingsManager.allowInventoryInteractionsDuringConversations && !KickStarter.settingsManager.allowGameplayDuringConversations)
			{
				return string.Empty;
			}

			string prefix = GetLabelPrefix (languageNumber, cursorID);
			string hotspotName = GetName (languageNumber);
			if (canBeLowerCase && !string.IsNullOrEmpty (prefix))
			{
				hotspotName = hotspotName.ToLower ();
			}

			return AdvGame.CombineLanguageString (prefix, hotspotName, languageNumber);
		}


		/**
		 * Recalculates the alpha value of the Hotspot's icon.
		 */
		public void UpdateIcon ()
		{
			CanDisplayHotspotIcon ();
		}


		/**
		 * <summary>Sets the layer of the Hotspot according to whether or not it is within the proximity of a Hotspot detector.</summary>
		 * <param name = "detectHotspot">The DetectHotspots component to check the proximity against</param>
		 */
		public void UpdateProximity (DetectHotspots detectHotspots)
		{
			if (detectHotspots == null) return;

			tooFarAway = !detectHotspots.IsHotspotInTrigger (this);
			if (tooFarAway)
			{
				PlaceOnDistantLayer ();
			}
			else
			{
				PlaceOnHotspotLayer ();
			}
		}


		/**
		 * <summary>Sets the layer of the Hotspot according to whether or not it has a "Use" interaction for the currently-selected cursor</summary>
		 * <returns>True if the Hotspot is on the default layer, False if not</returns>
		 */
		public bool UpdateUnhandledVisibility ()
		{
			if (KickStarter.runtimeInventory.SelectedItem != null)
			{
				if (!HasEnabledInvInteraction (KickStarter.runtimeInventory.SelectedItem.id))
				{
					PlaceOnDistantLayer ();
					return false;
				}
			}
			else
			{
				if (!HasEnabledUseInteraction (KickStarter.playerCursor.GetSelectedCursorID ()))
				{
					PlaceOnDistantLayer ();
					return false;
				}
			}

			PlaceOnHotspotLayer ();
			return true;
		}


		/**
		 * <summary>Shows or hides the Hotspot's associated icon, provided that the Settings Manager's hotspotIconDisplay = HotspotIconDisplay.ViaScriptOnly</summary>
		 * <param name = "makeVisible">If True, the icon will be shown. If false, the icon will be hidden</param>
		 * <param name = "speed">The speed at which to show or hide the icon. If <=0, the transition will be instantaneous.</param>
		 */
		public void SetIconVisibility (bool makeVisible, float speed = 5f)
		{
			manuallyShowIcon = makeVisible;
			manualShowIconSpeed = speed;
		}
		

		/**
		 * <summary>Gets the Button that represents the first-available "Use" interaction.</summary>
		 * <returns>The Button that represents the first-available "Use" interaction</returns>
		 */
		public Button GetFirstUseButton ()
		{
			foreach (Button button in useButtons)
			{
				if (button != null && !button.isDisabled)
				{
					return button;
				}
			}
			return null;
		}


		/**
		 * <summary>Gets the ID of the icon that represents the first-available "Use" interaction.</summary>
		 * <returns>The ID of the icon that represents the first-available "Use" interaction. If no appropriate interaction is found, -1 is returned</returns>
		 */
		public int GetFirstUseIcon ()
		{
			foreach (Button button in useButtons)
			{
				if (button != null && !button.isDisabled)
				{
					return button.iconID;
				}
			}
			return -1;
		}


		/**
		 * <summary>Gets the Button that represents the first-available "Use" interaction associated with a given interaction icon.</summary>
		 * <param name = "iconID">The ID number of the associated interaction icon, as defined in the Cursor Manager.</param>
		 * <returns>The Button that represents the first-available "Use" interaction associated with a given interaction ID.</returns>
		 */
		public Button GetUseButton (int iconID)
		{
			foreach (Button button in useButtons)
			{
				if (button != null && button.iconID == iconID)
				{
					return button;
				}
			}
			return null;
		}


		/**
		 * <summary>Gets the Button that represents the first-available "Inventory" interaction associated with a given inventory item.</summary>
		 * <param name = "invID">The ID number of the associated inventory item, as defined in the Inventory Manager.</param>
		 * <returns>The Button that represents the first-available "Inventory" interaction associated with a given inventory item.</returns>
		 */
		public Button GetInvButton (int invID)
		{
			foreach (Button button in invButtons)
			{
				if (button != null && button.invID == invID)
				{
					return button;
				}
			}
			return null;
		}


		/**
		 * <summary>Gets the interaction type of a given Button, provided it is defined by the Hotspot</summary>
		 * <param name = "_button">The Button to check</param>
		 * <returns>The Button's interaction type. If the Button is null or cannot be found on the Hotspot, HotspotInteractionType.NotFound will be returned</param>
		 */
		public HotspotInteractionType GetButtonInteractionType (Button _button)
		{
			if (_button != null)
			{
				if (lookButton == _button)
				{
					return HotspotInteractionType.Examine;
				}

				if (unhandledInvButton == _button)
				{
					return HotspotInteractionType.UnhandledInventory;
				}

				if (unhandledUseButton == _button)
				{
					return HotspotInteractionType.UnhandledUse;
				}

				foreach (Button button in useButtons)
				{
					if (button != null && _button == button)
					{
						return HotspotInteractionType.Use;;
					}
				}

				foreach (Button button in invButtons)
				{
					if (button != null && _button == button)
					{
						return HotspotInteractionType.Inventory;;
					}
				}
			}
			return HotspotInteractionType.NotFound;
		}


		/**
		 * <summary>Gets the index number of the Button that represents the first-available "Use" interaction.</summary>
		 * <returns>The index number of the Button that represents the first-available "Use" interaction</returns>
		 */
		public int FindFirstEnabledInteraction ()
		{
			if (useButtons != null && useButtons.Count > 0)
			{
				for (int i=0; i<useButtons.Count; i++)
				{
					if (!useButtons[i].isDisabled)
					{
						return i;
					}
				}
			}
			return 0;
		}


		/**
		 * <summary>Enables the Hotspot.</summary>
		 */
		public void TurnOn ()
		{
			TurnOn (true);
		}


		/**
		 * <summary>Updates the enabled state of one of the Hotspot's Buttons</summary>
		 * <param name="button">The Button to update</param>
		 * <param name="isEnabled">If True, the Button will be enabled. If False, it will be disabled</param>
		 */
		public void SetButtonState (Button button, bool isEnabled)
		{
			if (button != null && GetButtonInteractionType (button) != HotspotInteractionType.NotFound)
			{
				bool newDisabledState = !isEnabled;
				if (button.isDisabled != newDisabledState)
				{
					button.isDisabled = newDisabledState;

					KickStarter.eventManager.Call_OnHotspotSetInteractionState (this, button, isEnabled);
				}
			}
		}


		/**
		 * <summary>Enables the Hotspot.</summary>
		 * <param name = "manualSet">If True, then the Hotspot will be considered 'On" when saving</param>
		 */
		public virtual void TurnOn (bool manualSet)
		{
			if (tooFarAway)
			{
				gameObject.layer = LayerMask.NameToLayer (KickStarter.settingsManager.distantHotspotLayer);
			}
			else
			{
				gameObject.layer = LayerMask.NameToLayer (KickStarter.settingsManager.hotspotLayer);
			}

			if (manualSet)
			{
				if (!isOn && KickStarter.eventManager)
				{
					KickStarter.eventManager.Call_OnTurnHotspot (this, true);
				}

				isOn = true;

				if (KickStarter.mainCamera)
				{
					LimitToActiveCamera (KickStarter.mainCamera.attachedCamera);
				}
			}
		}


		/**
		 * <summary>Disables the Hotspot.</summary>
		 */
		public void TurnOff ()
		{
			TurnOff (true);
		}


		/**
		 * <summary>Disables the Hotspot.</summary>
		 * <param name = "manualSet">If True, then the Hotspot will be considered 'Off" when saving</param>
		 */
		public virtual void TurnOff (bool manualSet)
		{
			gameObject.layer = LayerMask.NameToLayer (KickStarter.settingsManager.deactivatedLayer);

			if (manualSet)
			{
				if (isOn && KickStarter.eventManager)
				{
					KickStarter.eventManager.Call_OnTurnHotspot (this, false);
				}

				isOn = false;

				if (KickStarter.player && KickStarter.player.hotspotDetector)
				{
					KickStarter.player.hotspotDetector.ForceRemoveHotspot (this);
				}
			}
		}
		

		/**
		 * <summary>Checks if the Hotspot is enabled or not.</summary>
		 * <returns>True if the Hotspot is enabled. If the Hotspot is not active only because its limitToCamera is not active, then True will be returned.</returns>
		 */
		public virtual bool IsOn ()
		{
			if (this == null || gameObject == null) return false;

			if (gameObject.layer == LayerMask.NameToLayer (KickStarter.settingsManager.deactivatedLayer) && !isOn)
			{
				return false;
			}

			return true;
		}


		/**
		 * <summary>Checks if the Player is within the Hotspot's interactableBoundary, if assigned.</summary>
		 * <returns>True if the Player is within the Hotspot's interactableBoundary, if assigned.  If no InteractableBoundary is assigned, or there is no Player, then True will be returned.</returns>
		 */
		public bool PlayerIsWithinBoundary ()
		{
			if (interactiveBoundary == null || KickStarter.player == null)
			{
				return true;
			}

			return interactiveBoundary.PlayerIsPresent;
		}
		

		/** Selects the Hotspot. */
		public void Select ()
		{
			KickStarter.eventManager.Call_OnChangeHotspot (this, true);

			if (highlight && highlight.highlightWhenSelected)
			{
				highlight.HighlightOn ();
			}
		}
		

		/**
		 * De-selects the Hotspot.
		 */
		public void Deselect ()
		{
			KickStarter.eventManager.Call_OnChangeHotspot (this, false);

			if (highlight)
			{
				highlight.HighlightOff ();
			}
		}

		
		/**
		 * De-selects the Hotspot instantly.
		 */
		public void DeselectInstant ()
		{
			KickStarter.eventManager.Call_OnChangeHotspot (this, false);
			
			if (highlight)
			{
				highlight.HighlightOffInstant ();
			}
		}


		/** Shows any Menus with appearType = AppearType.OnInteraction, linking any Interactino icons in it to the Hotspot */
		public void ShowInteractionMenus ()
		{
			if (KickStarter.playerMenus)
			{
				KickStarter.playerMenus.EnableInteractionMenus (this);
			}
		}


		/** 
		 * <summary>Turns on a specific Menu, linking any Interaction icons in it to the Hotspot. The Menu doesn't need to have an appearType of AppearType.OnInteraction.</summary>
		 * <param name = "menu">The Menu to turn on</param>
		 * <param name = "includeInventoryItems">If True, and supported, then inventory items associcated with the Hotspot's interactions will be included as well</param>
		 */
		public void ShowInteractionMenu (Menu menu, bool includeInventoryItems)
		{
			menu.MatchInteractions (this, includeInventoryItems);
			menu.TurnOn ();
		}
		

		/**
		 * <summary>Checks if oneClick = True, and the Hotspot has at least one "Use" interaction defined.</summary>
		 * <returns>True if oneClick = True, and the Hotspot has at least one "Use" interaction defined.</summmary>
		 */
		public bool IsSingleInteraction ()
		{
			if (oneClick && provideUseInteraction && useButtons != null && GetFirstUseButton () != null)
			{
				return true;
			}
			return false;
		}


		/**
		 * <summary>Checks if the Hotspot has an active interaction for a given inventory item, or a generic unhandled inventory interaction.</summary>
		 * <param name = "invItem">The inventory item to check for</param>
		 * <returns>True if the Hotspot has an active interaction for the inventory item, or a generic unhandled inventory interaction.</returns>
		 */
		public bool HasInventoryInteraction (InvItem invItem)
		{
			if (invItem != null)
			{
				if (provideUnhandledInvInteraction && unhandledInvButton != null && !unhandledInvButton.isDisabled)
				{
					return true;
				}

				if (provideInvInteraction && invButtons != null && invButtons.Count > 0)
				{
					for (int i=0; i<invButtons.Count; i++)
					{
						if (!invButtons[i].isDisabled && invButtons[i].invID == invItem.id)
						{
							return true;
						}
					}
				}
			}
			return false;
		}


		/**
		 * <summary>Gets the position of the Hotspot's icon, in Screen Space.</summary>
		 * <returns>The position of the Hotspot's icon, in Screen Space.</returns>
		 */
		public Vector2 GetIconScreenPosition ()
		{
			Vector3 screenPosition = KickStarter.CameraMain.WorldToScreenPoint (GetIconPosition ());
			return new Vector3 (screenPosition.x, screenPosition.y);
		}
		

		/**
		 * <summary>Gets the position of the Hotspot's icon</summary>
		 * <param = "inLocalSpace">If True, the position returned will be relative to the centre of the Hotspot's transform, rather than the scene's origin</param>
		 * <returns>The position of the Hotspot's icon</returns>
		 */
		public virtual Vector3 GetIconPosition (bool inLocalSpace = false)
		{
			Vector3 worldPoint = Transform.position;

			if (centrePoint && centrePointOverrides != CentrePointOverrides.FacingPositionOnly)
			{
				if (inLocalSpace)
				{
					return (centrePoint.position - Transform.position);
				}
				return centrePoint.position;
			}
			
			if (_collider)
			{
				worldPoint = _collider.bounds.center;
			}
			else if (_collider2D)
			{
				worldPoint = _collider2D.bounds.center;
			}

			if (inLocalSpace)
			{
				return worldPoint - Transform.position;
			}
			return worldPoint;
		}


		/**
		 * <summary>The Hotspot's facing position</summary>
		 * <param = "inLocalSpace">If True, the position returned will be relative to the centre of the Hotspot's transform, rather than the scene's origin</param>
		 * <returns>The Hotspot's facing position</returns>
		 */
		public virtual Vector3 GetFacingPosition (bool inLocalSpace = false)
		{
			Vector3 worldPoint = Transform.position;

			if (centrePoint && centrePointOverrides != CentrePointOverrides.IconPositionOnly)
			{
				if (inLocalSpace)
				{
					return (centrePoint.position - Transform.position);
				}
				return centrePoint.position;
			}

			if (_collider)
			{
				worldPoint = _collider.bounds.center;
			}
			else if (_collider2D)
			{
				worldPoint = _collider2D.bounds.center;
			}

			if (inLocalSpace)
			{
				return worldPoint - Transform.position;
			}
			return worldPoint;
		}


		/**
		 * Clears the Hotspot's internal 'use' icon, as used when the Hotspot is highlighted.
		 */
		public void ResetMainIcon ()
		{
			mainIcon = null;
		}


		/**
		 * <summary>Gets the previous interaction index.</summary>
		 * <param name = "i">The current interaction index</param>
		 * <param name = "numInvInteractions">The number of relevant "Inventory" interactions that match the current cursor</param>
		 * <returns>The previous interaction index</returns>
		 */
		public int GetPreviousInteraction (int i, int numInvInteractions)
		{
			if (i > useButtons.Count && numInvInteractions > 0)
			{
				return (i-1);
			}
			else if (i == 0)
			{
				return FindLastEnabledInteraction (numInvInteractions);
			}
			else if (i <= useButtons.Count)
			{
				i --;
				while (i > 0 && useButtons [i].isDisabled)
				{
					i --;
				}

				if (i < 0)
				{
					return FindLastEnabledInteraction (numInvInteractions);
				}
				else
				{
					if (i == 0 && useButtons.Count > 0 && useButtons[0].isDisabled)
					{
						return FindLastEnabledInteraction (numInvInteractions);
					}
					return i;
				}
			}

			return (i-1);
		}


		public int GetPreviousInteraction (int i)
		{
			return GetPreviousInteraction (i, GetMatchingInvInteractionData (true).NumMatchingInteractions);
		}


		public int GetActiveInvButtonID ()
		{
			int interactionIndex = KickStarter.playerInteraction.InteractionIndex;

			if (KickStarter.settingsManager.SelectInteractionMethod () == SelectInteractions.CyclingCursorAndClickingHotspot)
			{
				if (interactionIndex >= useButtons.Count)
				{
					int matchingInvIndex = interactionIndex - useButtons.Count;
					if (matchingInvIndex < GetMatchingInvInteractionData (false).NumMatchingInteractions)
					{
						int invButtonIndex = GetMatchingInvInteractionData (false).GetInvInteractionIndex (matchingInvIndex);
						Button invButton = invButtons[invButtonIndex];
						return invButton.invID;
					}
				}
			}
			else
			{
				// Cycle menus

				int matchingInvIndex = interactionIndex - useButtons.Count;
				if (matchingInvIndex >= 0 && GetMatchingInvInteractionData (false).NumMatchingInteractions > matchingInvIndex)
				{
					int invButtonIndex = GetMatchingInvInteractionData (false).GetInvInteractionIndex (matchingInvIndex);
					if (invButtons.Count > invButtonIndex)
					{
						Button invButton = invButtons[invButtonIndex];
						return invButton.invID;
					}
				}
			}
			return -1;
		}


		public void RestoreInteraction ()
		{
			if (!KickStarter.settingsManager.cycleInventoryCursors && InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance))
			{
				return;
			}

			GetMatchingInvInteractionData (true);

			switch (KickStarter.settingsManager.whenReselectHotspot)
			{
				case WhenReselectHotspot.ResetIcon:
					KickStarter.playerInteraction.InteractionIndex = lastInteractionIndex = 0;
					return;

				case WhenReselectHotspot.RestoreHotspotIcon:
					KickStarter.playerInteraction.InteractionIndex = lastInteractionIndex;
					if (!KickStarter.settingsManager.cycleInventoryCursors && GetActiveInvButtonID () >= 0)
					{
						KickStarter.playerInteraction.InteractionIndex = -1;
						return;
					}
					else
					{
						int invID = GetActiveInvButtonID ();
						if (invID >= 0)
						{
							KickStarter.runtimeInventory.SelectItemByID (invID, SelectItemMode.Use);
						}
						else
						{
							KickStarter.runtimeInventory.SetNull ();
						}
					}
					break;

				default:
					break;
			}

			KickStarter.playerInteraction.InteractionIndex = lastInteractionIndex;
		}


		/**
		 * <summary>Gets the Hotspot's current display name.</summary>
		 * <param name = "languageNumber">The index number of the game's current language</param>
		 * <returns>The Hotspot's current display name</returns>
		 */
		public string GetName (int languageNumber)
		{
			string newName = gameObject.name;
			if (!string.IsNullOrEmpty (hotspotName))
			{
				newName = hotspotName;
			}

			if (languageNumber > 0)
			{
				return KickStarter.runtimeLanguages.GetTranslation (newName, displayLineID, languageNumber, GetTranslationType (0));
			}

			return newName;
		}


		/**
		 * <summary>Renames the Hotspot mid-game.</summary>
		 * <param name = "newName">The new name of the Hotspot</param>
		 * <param name = "_lineID">The translation ID number assocated with the new name, as set by SpeechManager</param>
		 */
		public void SetName (string newName, int _lineID)
		{
			hotspotName = newName;

			if (_lineID >= 0)
			{
				displayLineID = _lineID;
			}
			else
			{
				displayLineID = lineID;
			}
		}


		/**
		 * <summary>Checks if the Hotspot has at least one "Use" interaction defined.</summary>
		 * <returns>True if the Hotspot has at least one "Use" interaction defined.</returns>
		 */
		public bool HasContextUse ()
		{
			if ((oneClick || KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.ContextSensitive) && provideUseInteraction && useButtons != null && GetFirstUseButton () != null)
			{
				return true;
			}
			
			return false;
		}
		

		/**
		 * <summary>Checks if the Hotspot has at least one "Examine" interaction defined.</summary>
		 * <returns>True if the Hotspot has at least one "Examine" interaction defined.</returns>
		 */
		public bool HasContextLook ()
		{
			if (provideLookInteraction && lookButton != null && !lookButton.isDisabled)
			{
				return true;
			}
			
			return false;
		}


		/**
		 * <summary>Gets the next interaction index.</summary>
		 * <param name = "i">The current interaction index</param>
		 * <param name = "numInvInteractions">The number of relevant "Inventory" interactions that match the current cursor</param>
		 * <returns>The next interaction index</returns>
		 */
		public int GetNextInteraction (int i, int numInvInteractions)
		{
			if (i < useButtons.Count)
			{
				if (IsSingleInteraction ())
				{
					i = useButtons.Count;
				}
				else
				{
					i ++;
				}

				while (i<useButtons.Count && useButtons[i].isDisabled)
				{
					i++;
				}

				if (i >= useButtons.Count + numInvInteractions)
				{
					return FindFirstEnabledInteraction ();
				}
				else
				{

					return i;
				}
			}
			else if (i >= useButtons.Count - 1 + numInvInteractions)
			{
				return FindFirstEnabledInteraction ();
			}

			return (i+1);
		}


		public int GetNextInteraction (int i)
		{
			return GetNextInteraction (i, GetMatchingInvInteractionData (true).NumMatchingInteractions);
		}


		public MatchingInvInteractionData GetMatchingInvInteractionData (bool rebuild)
		{
			if (rebuild || matchingInvInteractionData == null)
			{
				matchingInvInteractionData = new MatchingInvInteractionData (this);
			}
			return matchingInvInteractionData;
		}

		#endregion


		#region ProtectedFunctions

		protected void OnSwitchCamera (_Camera oldCamera, _Camera newCamera, float transitionTime)
		{
			if (limitToCamera == null) return;

			LimitToActiveCamera (newCamera);
		}


		protected void LimitToActiveCamera (_Camera _camera)
		{
			if (limitToCamera && _camera)
			{
				if (_camera == limitToCamera && isOn)
				{
					TurnOn (false);
				}
				else
				{
					TurnOff (false);
				}
			}
		}


		protected void FindFirstInteractionIndex ()
		{
			lastInteractionIndex = 0;

			foreach (Button button in useButtons)
			{
				if (!button.isDisabled)
				{
					lastInteractionIndex = useButtons.IndexOf (button);
					return;
				}
			}
		}


		protected void PlaceOnDistantLayer ()
		{
			if (gameObject.layer == LayerMask.NameToLayer (KickStarter.settingsManager.hotspotLayer))
			{
				gameObject.layer = LayerMask.NameToLayer (KickStarter.settingsManager.distantHotspotLayer);
			}
		}


		protected void PlaceOnHotspotLayer ()
		{
			if (gameObject.layer == LayerMask.NameToLayer (KickStarter.settingsManager.distantHotspotLayer))
			{
				gameObject.layer = LayerMask.NameToLayer (KickStarter.settingsManager.hotspotLayer);
			}
		}


		protected bool CanDisplayHotspotIcon ()
		{
			if (gameObject.layer != LayerMask.NameToLayer (KickStarter.settingsManager.deactivatedLayer))
			{
				if (KickStarter.CameraMain == null) return false;

				Vector3 direction = (Transform.position - KickStarter.CameraMainTransform.position);
				if (Vector3.Angle (direction, KickStarter.CameraMainTransform.forward) > 90f)
				{
					iconAlpha = 0f;
					return false;
				}
				
				if (SceneSettings.CameraPerspective != CameraPerspective.TwoD && KickStarter.settingsManager.occludeIcons)
				{
					// Is icon occluded?
					Ray ray = new Ray (KickStarter.CameraMainTransform.position, GetIconPosition () - KickStarter.CameraMainTransform.position);
					RaycastHit hit;
					if (Physics.Raycast (ray, out hit, KickStarter.settingsManager.hotspotRaycastLength, 1 << LayerMask.NameToLayer (KickStarter.settingsManager.hotspotLayer)))
					{
						if (hit.collider.gameObject != this.gameObject)
						{
							iconAlpha = 0f;
							return false;
						}
					}
				}

				if (!KickStarter.stateHandler.IsInGameplay ())
				{
					iconAlpha = 0f;
					return false;
				}
				else if (KickStarter.playerMenus.IsInteractionMenuOn () && KickStarter.settingsManager.hideIconUnderInteractionMenu)
				{
					iconAlpha = iconAlphaLerp.Update (iconAlpha, 0f, 5f);
				}
				else if (KickStarter.settingsManager.hotspotIconDisplay == HotspotIconDisplay.ViaScriptOnly)
				{
					if (manualShowIconSpeed > 0f)
					{
						iconAlpha = iconAlphaLerp.Update (iconAlpha, (manuallyShowIcon) ? 1f : 0f, manualShowIconSpeed);
					}
					else
					{
						iconAlpha = (manuallyShowIcon) ? 1f : 0f;
					}
				}
				else if (KickStarter.settingsManager.hotspotIconDisplay == HotspotIconDisplay.OnlyWhenHighlighting ||
				         KickStarter.settingsManager.hotspotIconDisplay == HotspotIconDisplay.OnlyWhenFlashing)
				{
					if (highlight)
					{
						if (KickStarter.settingsManager.hotspotIconDisplay == HotspotIconDisplay.OnlyWhenHighlighting)
						{
							iconAlpha = highlight.GetHighlightAlpha ();
						}

						else
						{
							iconAlpha = highlight.GetFlashAlpha (iconAlpha);
						}
					}
					else
					{
						ACDebug.LogWarning ("Cannot display correct Hotspot Icon alpha on " + name + " because it has no associated Highlight object.", gameObject);
					}
				}
				else if (KickStarter.settingsManager.hotspotIconDisplay == HotspotIconDisplay.Always)
				{
					iconAlpha = 1f;
				}
				else
				{
					iconAlpha = 0f;
				}
				return true;
			}
			else
			{
				iconAlpha = 0f;
				return false;
			}
		}


		protected void GetMainIcon ()
		{
			if (mainIcon != null)
			{
				return;
			}

			if (KickStarter.cursorManager == null)
			{
				return;
			}
			
			if (provideUseInteraction && useButton != null && useButton.iconID >= 0 && !useButton.isDisabled)
			{
				mainIcon = new CursorIcon ();
				mainIcon.Copy (KickStarter.cursorManager.GetCursorIconFromID (useButton.iconID), true);
				return;
			}
			
			if (provideLookInteraction && lookButton != null && lookButton.iconID >= 0 && !lookButton.isDisabled)
			{
				mainIcon = new CursorIcon ();
				mainIcon.Copy (KickStarter.cursorManager.GetCursorIconFromID (lookButton.iconID), true);
				return;
			}
			
			if (provideUseInteraction && useButtons != null && useButtons.Count > 0)
			{
				for (int i=0; i<useButtons.Count; i++)
				{
					if (!useButtons[i].isDisabled)
					{
						mainIcon = new CursorIcon ();
						mainIcon.Copy (KickStarter.cursorManager.GetCursorIconFromID (useButtons[i].iconID), true);
						return;
					}
				}
			}
		}


		protected int FindLastEnabledInteraction (int numInvInteractions)
		{
			if (numInvInteractions > 0)
			{
				if (useButtons != null)
				{
					return (useButtons.Count - 1 + numInvInteractions);
				}
				return (numInvInteractions - 1);
			}

			if (useButtons != null && useButtons.Count > 0)
			{
				for (int i=useButtons.Count-1; i>=0; i--)
				{
					if (!useButtons[i].isDisabled)
					{
						return i;
					}
				}
			}
			return 0;
		}


		protected bool HasEnabledUseInteraction (int _iconID)
		{
			if (_iconID >= 0)
			{
				for (int i=0; i<useButtons.Count; i++)
				{
					if (useButtons[i].iconID == _iconID && !useButtons[i].isDisabled)
					{
						return true;
					}
				}
			}
			return false;
		}


		protected bool HasEnabledInvInteraction (int _itemID)
		{
			if (_itemID >= 0)
			{
				for (int i = 0; i < invButtons.Count; i++)
				{
					if (useButtons[i].invID == _itemID && !useButtons[i].isDisabled)
					{
						return true;
					}
				}
			}
			return false;
		}


		protected int GetNumInteractions (int numInvInteractions)
		{
			int num = 0;
			foreach (Button _button in useButtons)
			{
				if (!_button.isDisabled)
				{
					num ++;
				}
			}
			return (num + numInvInteractions);
		}


		protected string GetLabelPrefix (int languageNumber = 0, int cursorID = -1)
		{
			int interactionIndex = KickStarter.playerInteraction.InteractionIndex;

			bool isOverride = (cursorID >= 0);
			if (!isOverride)
			{
				if (cursorID == -1 && IsSingleInteraction () && !InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance) && KickStarter.settingsManager.interactionMethod != AC_InteractionMethod.ContextSensitive)
				{
					cursorID = GetFirstUseIcon ();
				}
				else
				{
					cursorID = KickStarter.playerCursor.GetSelectedCursorID ();
				}
			}

			string label = string.Empty;
			if (InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance) && 
				(KickStarter.cursorManager.inventoryHandling == InventoryHandling.ChangeHotspotLabel || KickStarter.cursorManager.inventoryHandling == InventoryHandling.ChangeCursorAndHotspotLabel))
			{
				label = KickStarter.runtimeInventory.SelectedInstance.GetHotspotPrefixLabel (languageNumber, true);
			}
			else
			{
				if (KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.ChooseHotspotThenInteraction)
				{
					if (KickStarter.settingsManager.selectInteractions == SelectInteractions.CyclingMenuAndClickingHotspot)
					{
						if (interactionIndex >= 0 && KickStarter.playerMenus.IsInteractionMenuOn ())
						{
							if (interactionIndex >= useButtons.Count)
							{
								// Use Inventory item on Hotspot
								int itemIndex = interactionIndex - useButtons.Count;
								if (invButtons.Count > itemIndex)
								{
									InvInstance invInstance = KickStarter.runtimeInventory.GetInstance (invButtons[itemIndex].invID);
									if (InvInstance.IsValid (invInstance))
									{
										invInstance.SelectItemMode = invButtons[itemIndex].selectItemMode;
									}
								}
							}
						}
					}
				}

				if (KickStarter.cursorManager.addHotspotPrefix)
				{
					switch (KickStarter.settingsManager.interactionMethod)
					{
						case AC_InteractionMethod.ContextSensitive:
							if (provideUseInteraction && !InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance))
							{
								Button _button = GetFirstUseButton ();
								if (_button != null)
								{
									label = KickStarter.cursorManager.GetLabelFromID (_button.iconID, languageNumber);
								}
							}
							break;

						case AC_InteractionMethod.ChooseHotspotThenInteraction:
							if (KickStarter.settingsManager.selectInteractions == SelectInteractions.CyclingCursorAndClickingHotspot ||
							KickStarter.settingsManager.selectInteractions == SelectInteractions.ClickingMenu)
							{
								label = KickStarter.cursorManager.GetLabelFromID (cursorID, languageNumber);
							}
							else if (KickStarter.settingsManager.selectInteractions == SelectInteractions.CyclingMenuAndClickingHotspot)
							{
								if (interactionIndex >= 0 && KickStarter.playerMenus.IsInteractionMenuOn ())
								{
									if (interactionIndex < useButtons.Count)
									{
										label = KickStarter.cursorManager.GetLabelFromID (useButtons[interactionIndex].iconID, languageNumber);
									}
									else
									{
										// Inventory item
										int itemIndex = interactionIndex - useButtons.Count;
										if (invButtons.Count > itemIndex)
										{
											InvInstance invInstance = KickStarter.runtimeInventory.GetInstance (invButtons[itemIndex].invID);
											if (InvInstance.IsValid (invInstance))
											{
												label = invInstance.GetHotspotPrefixLabel (languageNumber, true);
											}
										}
									}
								}
								else if (IsSingleInteraction ())
								{
									if (provideUseInteraction && !InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance))
									{
										Button _button = GetFirstUseButton ();
										if (_button != null)
										{
											label = KickStarter.cursorManager.GetLabelFromID (_button.iconID, languageNumber);
										}
									}
								}
							}
							break;

						case AC_InteractionMethod.ChooseInteractionThenHotspot:
						case AC_InteractionMethod.CustomScript:
							label = KickStarter.cursorManager.GetLabelFromID (cursorID, languageNumber);
							break;

						default:
							break;
					}
				}
			}

			if (!isOverride && cursorID == -1 && !InvInstance.IsValid (KickStarter.runtimeInventory.SelectedInstance) && KickStarter.cursorManager.addWalkPrefix && !KickStarter.playerMenus.IsInteractionMenuOn ())
			{
				// Only show "Walk to" for Hotspots
				if (KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.ContextSensitive && GetFirstUseButton () != null)
				{
					// Ignore in this case
				}
				else
				{
					label = KickStarter.runtimeLanguages.GetTranslation (KickStarter.cursorManager.walkPrefix.label, KickStarter.cursorManager.walkPrefix.lineID, languageNumber, KickStarter.cursorManager.walkPrefix.GetTranslationType (0));

				}
			}

			return label;
		}

		#endregion


		#if UNITY_EDITOR

		protected void OnDrawGizmos ()
		{
			if (KickStarter.sceneSettings && KickStarter.sceneSettings.visibilityHotspots && UnityEditor.Selection.activeGameObject != gameObject)
			{
				DrawGizmos ();
			}
		}
		
		
		protected void OnDrawGizmosSelected ()
		{
			DrawGizmos ();
		}


		protected void DrawGizmos ()
		{
			if (drawGizmos)
			{
				Color gizmoColor = ACEditorPrefs.HotspotGizmoColor;

				PolygonCollider2D polygonCollider2D = GetComponent <PolygonCollider2D>();
				if (polygonCollider2D)
				{
					AdvGame.DrawPolygonCollider (transform, polygonCollider2D, gizmoColor);
				}
				else
				{
					MeshCollider meshCollider = GetComponent <MeshCollider>();
					if (meshCollider)
					{
						AdvGame.DrawMeshCollider (transform, meshCollider.sharedMesh, gizmoColor);
					}
					else
					{
						SphereCollider sphereCollider = GetComponent <SphereCollider>();
						if (sphereCollider)
						{
							AdvGame.DrawSphereCollider (transform, sphereCollider, gizmoColor);
						}
						else
						{
							CapsuleCollider capsuleCollider = GetComponent <CapsuleCollider>();
							if (capsuleCollider)
							{
								AdvGame.DrawCapsule (transform, capsuleCollider.center, capsuleCollider.radius, capsuleCollider.height, gizmoColor);
							}
							else
							{
								CharacterController characterController = GetComponent <CharacterController>();
								if (characterController)
								{
									AdvGame.DrawCapsule (transform, characterController.center, characterController.radius, characterController.height, gizmoColor);
								}
								else
								{
									AdvGame.DrawCubeCollider (transform, gizmoColor);
								}
							}
						}
					}
				}
			}
		}


		public int GetInventoryReferences (int invID)
		{
			int numFound = 0;
			foreach (Button invButton in invButtons)
			{
				if (invButton.invID == invID)
				{
					numFound ++;
				}
			}
			return numFound;
		}

		#endif


		#region GetSet

		/** A cache of the Hotspot's transform component */
		public Transform Transform
		{
			get
			{
				if (_transform == null) _transform = transform;
				return _transform;
			}
		}

		#endregion


		#region ITranslatable

		public string GetTranslatableString (int index)
		{
			if (!string.IsNullOrEmpty (hotspotName))
			{
				return hotspotName;
			}
			return name;
		}


		public int GetTranslationID (int index)
		{
			return lineID;
		}


		public AC_TextType GetTranslationType (int index)
		{
			return AC_TextType.Hotspot;
		}


		#if UNITY_EDITOR

		public void UpdateTranslatableString (int index, string updatedText)
		{
			hotspotName = updatedText;
		}


		public int GetNumTranslatables ()
		{
			return 1;
		}


		public bool HasExistingTranslation (int index)
		{
			return (lineID > -1);
		}


		public void SetTranslationID (int index, int _lineID)
		{
			lineID = _lineID;
		}


		public string GetOwner (int index)
		{
			return string.Empty;
		}


		public bool OwnerIsPlayer (int index)
		{
			return false;
		}


		public bool CanTranslate (int index)
		{
			return (!string.IsNullOrEmpty (name) || !string.IsNullOrEmpty (hotspotName));
		}

		#endif

		#endregion


		#if UNITY_EDITOR

		public bool ReferencesAsset (ActionListAsset actionListAsset)
		{
			if (interactionSource == AC.InteractionSource.AssetFile)
			{
				if (provideUseInteraction)
				{
					foreach (Button _useButton in useButtons)
					{
						if (_useButton.assetFile == actionListAsset) return true;
					}
				}

				if (KickStarter.settingsManager == null || KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.ContextSensitive || KickStarter.settingsManager.interactionMethod == AC_InteractionMethod.CustomScript)
				{
					if (provideLookInteraction)
					{
						if (lookButton.assetFile == actionListAsset) return true;
					}
				}

				if (provideInvInteraction)
				{
					foreach (Button invButton in invButtons)
					{
						if (invButton.assetFile == actionListAsset) return true;
					}
				}

				if (provideUnhandledInvInteraction)
				{
					if (unhandledInvButton.assetFile == actionListAsset) return true;
				}
			}
			return false;
		}

		#endif

	}

}