/*
 *
 *	Adventure Creator
 *	by Chris Burton, 2013-2021
 *	
 *	"Char.cs"
 * 
 *	This is the base class for both NPCs and the Player.
 *	It contains the functions needed for animation and movement.
 * 
 */

using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.AI;
using UnityEngine.Playables;
#if SalsaIsPresent
using CrazyMinnow.SALSA;
#endif

namespace AC
{

	/**
	 * The base class for both NPCs and the Player.
	 * It contains the functions needed for animation and movement.
	 */
	[HelpURL ("https://www.adventurecreator.org/scripting-guide/class_a_c_1_1_char.html")]
	public class Char : MonoBehaviour, ITranslatable
	{

		/** The character's state (Idle, Move, Decelarate, Custom) */
		public CharState charState;

		private AnimEngine animEngine;

		// Lerp instances
		private LerpUtils.FloatLerp turnFloatLerp = new LerpUtils.FloatLerp ();
		private LerpUtils.FloatLerp newAngleLerp = new LerpUtils.FloatLerp ();
		private LerpUtils.FloatLerp newAngleLinearLerp = new LerpUtils.FloatLerp (true);
		private LerpUtils.Vector2Lerp targetHeadAnglesLerp = new LerpUtils.Vector2Lerp ();
		private LerpUtils.FloatLerp headTurnWeightLerp = new LerpUtils.FloatLerp ();
		private LerpUtils.Vector2Lerp actualHeadAnglesLerp = new LerpUtils.Vector2Lerp ();
		private LerpUtils.FloatLerp nonFacingFactorLerp = new LerpUtils.FloatLerp ();
		private LerpUtils.FloatLerp wallReductionLerp = new LerpUtils.FloatLerp ();
		protected LerpUtils.FloatLerp moveSpeedLerp = new LerpUtils.FloatLerp ();
		private LerpUtils.Vector3Lerp exactPositionLerp = new LerpUtils.Vector3Lerp ();

		/** The animation engine enum (SpritesUnity, SpritesUnityComplex, Sprites2DToolkit, Legacy, Mecanim, Custom) */
		public AnimationEngine animationEngine = AnimationEngine.SpritesUnity;
		/** The class name of the AnimEngine ScriptableObject that animates the character, if animationEngine = AnimationEngine.Custom */
		public string customAnimationClass = "";
		/** How motion is controlled, if animationEngine = AnimationEngine.Custom (Automatic, JustTurning, Manual) */
		public MotionControl motionControl = MotionControl.Automatic;
		/** How talking animations are handled (Standard, CustomFace) */
		public TalkingAnimation talkingAnimation = TalkingAnimation.Standard;
		/** If True, using SpritesUnity animation, and talkingAnimation = TalkingAnimation.Standard, then the head animation (talking) will be handled on a non-root layer. */
		public bool separateTalkingLayer = false;

		/** The character's display name when speaking */
		public string speechLabel = "";
		/** The translation ID number of the Character's name, as generated by SpeechManager */
		public int lineID = -1;
		/** The translation ID number of the Hotspot's name, if it was changed mid-game */
		public int displayLineID = -1;
		/** The colour of the character's speech text */
		public Color speechColor = Color.white;
		/** The character's portrait graphic, used in MenuGraphic elements when speaking */
		public CursorIconBase portraitIcon = new CursorIconBase ();
		/** If True, speech text can use expression tokens to change the character's expression */
		public bool useExpressions;
		/** The character's available expressions, if useExpressions = True */
		public List<Expression> expressions = new List<Expression> ();
		/** The Transform at which to place Menus set to appear 'Above Speaking Character'. If this is not set, the placement will be set automatically. */
		public Transform speechMenuPlacement;
		/** If useExpressions = True, and the chosen animation engine allows for it, a Shapeable component can be mapped to expressions to allow for expression tokens to control blendshapes */
		public bool mapExpressionsToShapeable = false;
		/** The Shapeable group ID that controls expression blendshapes, if using AnimEngine_Legacy / AnimEngine_Mecanim */
		public int expressionGroupID;
		/** A speed tfactor to apply to the changing of blend-shapes based on lipsyncing */
		public float lipSyncBlendShapeSpeedFactor = 1f;
		private Expression currentExpression = null;

		protected Quaternion newRotation;
		private float prevHeight;
		private float heightChange;
		protected Transform _transform;
		protected Speech activeSpeech;

		// Lip sync variables

		private Shapeable shapeable = null;
		private LipSyncTexture lipSyncTexture = null;
		private List<LipSyncShape> lipSyncShapes = new List<LipSyncShape> ();

		/** If True, the character is currently lip-syncing */
		public bool isLipSyncing = false;
		/** The name of the Animator integer parameter set to the lip-syncing phoneme integer, if using AnimEngine_SpritesUnityComplex */
		public string phonemeParameter = "";
		/** The Shapeable group ID that controls phoneme blendshapes, if using AnimEngine_Legacy / AnimEngine_Mecanim */
		public int lipSyncGroupID;

		#if SalsaIsPresent
		private Salsa2D salsa2D;
		#endif

		// 3D variables

		/** The Transform to parent objects held in the character's left hand to */
		public Transform leftHandBone;
		/** The Transform to parent objects held in the character's right hand to */
		public Transform rightHandBone;
		private GameObject leftHandHeldObject;
		private GameObject rightHandHeldObject;

		private IKLimbController leftHandIKController = new IKLimbController ();
		private IKLimbController rightHandIKController = new IKLimbController ();
		
		// Legacy variables

		/** The "Idle" animation, if using AnimEngine_Legacy */
		public AnimationClip idleAnim;
		/** The "Walk" animation, if using AnimEngine_Legacy */
		public AnimationClip walkAnim;
		/** The "Run" animation, if using AnimEngine_Legacy */
		public AnimationClip runAnim;
		/** The "Talk" animation, if using AnimEngine_Legacy */
		public AnimationClip talkAnim;
		/** The "Turn left" animation, if using AnimEngine_Legacy */
		public AnimationClip turnLeftAnim;
		/** The "Turn right" animation, if using AnimEngine_Legacy */
		public AnimationClip turnRightAnim;
		/** The "Look left" animation, if using AnimEngine_Legacy */
		public AnimationClip headLookLeftAnim;
		/** The "Look right" animation, if using AnimEngine_Legacy */
		public AnimationClip headLookRightAnim;
		/** The "Look up" animation, if using AnimEngine_Legacy */
		public AnimationClip headLookUpAnim;
		/** The "Look down" animation, if using AnimEngine_Legacy */
		public AnimationClip headLookDownAnim;

		private Animation _animation;

		/** The "Upper body" bone Transform, used to isolate animations if using AnimEngine_Legacy */
		public Transform upperBodyBone;
		/** The "Left arm" bone Transform, used to isolate animations if using AnimEngine_Legacy */
		public Transform leftArmBone;
		/** The "Right arm" bone Transform, used to isolate animations if using AnimEngine_Legacy */
		public Transform rightArmBone;
		/** The "Neck" bone Transform, used to isolate animations if using AnimEngine_Legacy */
		public Transform neckBone;

		/** The default duration, in seconds, when crossfading standard animations, if using AnimEngine_Legacy */
		public float animCrossfadeSpeed = 0.2f;

		private Vector3 exactDestination;

		/** The layermask to use when Raycasting to determine if the character is grounded or not */
		public int groundCheckLayerMask = 1;

		// Mecanim variables

		/** The name of the Animator float parameter set to the movement speed, if using AnimEngine_Mecanim */
		public string moveSpeedParameter = "Speed";
		/** The name of the Animator float parameter set to the vertical movement speed, if using AnimEngine_Mecanim */
		public string verticalMovementParameter = "";
		/** The name of the Animator bool parameter set to the 'Is Grounded' check, is using AnimEngine_Mecanim */
		public string isGroundedParameter;
		/** The name of the Animator boolean parameter to set to 'True' when jumping, if using Mecanim animation (Player characters only) */
		public string jumpParameter = "";
		/** The name of the Animator float parameter set to the turning direction, if using AnimEngine_Mecanim */
		public string turnParameter = "";
		/** The name of the Animator bool parameter set to True while talking, if using AnimEngine_Mecanim */
		public string talkParameter = "IsTalking";
		/** The name of the Animator integer parameter set to the sprite direction (set by GetSpriteDirectionInt()), if using AnimEngine_SpritesUnityComplex */
		public string directionParameter = "Direction";
		/** The name of the Animator float parameter set to the facing angle, if using AnimEngine_SpritesUnityComplex */
		public string angleParameter = "Angle";
		/** The name of the Animator float parameter set to the head yaw, if using AnimEngine_Mecanim */
		public string headYawParameter = "";
		/** The name of the Animator float parameter set to the head pitch, if using AnimEngine_Mecanim */
		public string headPitchParameter = "";
		/** The name of the Animator integer parameter set to the active Expression ID number */
		public string expressionParameter = "";
		/** The factor by which the job of turning is left to Mecanim root motion, if using AnimEngine_Mecanim */
		public float rootTurningFactor = 0f;
		/** The Animator layer used to play head animations while talking, by AnimEngine_Mecanim / AnimEngine_SpritesUnity */
		public int headLayer = 1;
		/** The Animator layer used to play mouth animations while talking, by AnimEngine_Mecanim / AnimEngine_SpritesUnity */
		public int mouthLayer = 2;
		/** The Animator component, which will be assigned automatically if not set manually */
		public Animator customAnimator;
		private bool animatorIsOnRoot;
		/** The minimum angle between the character's current direction, and their intended direction, for spot-turning to be possible */
		public float turningAngleThreshold = 4f;


		#if UNITY_EDITOR
		public bool listExpectedAnimations;
		#endif

		// 2D variables

		private Animator _animator;

		/** If True, then the root object of a 2D, sprite-based character will rotate around the Z-axis. Otherwise, turning will be simulated and the actual rotation will be unaffected */
		public bool turn2DCharactersIn3DSpace = true;

		/** The sprite Transform, that's a child GameObject, used by AnimEngine_SpritesUnity / AnimEngine_SpritesUnityComplex / AnimEngine_Sprites2DToolkit */
		public Transform spriteChild;
		/** The method by which a sprite-based character should face the camera in 3D games (CameraRotation, CameraPosition) */
		public RotateSprite3D rotateSprite3D = RotateSprite3D.CameraFacingDirection;

		/** The name of the 'Idle' animation(s), without suffix, used by AnimEngine_SpritesUnity */
		public string idleAnimSprite = "idle";
		/** The name of the 'Walk' animation(s), without suffix, used by AnimEngine_SpritesUnity */
		public string walkAnimSprite = "walk";
		/** The name of the 'Run' animation(s), without suffix, used by AnimEngine_SpritesUnity */
		public string runAnimSprite = "run";
		/** The name of the 'Talk' animation(s), without suffix, used by AnimEngine_SpritesUnity */
		public string talkAnimSprite = "talk";

		/** If True, sprite-based characters will not resize along a SortingMap */
		public bool lockScale = false;
		/** A sprite-based character's scale, if lockScale = True */
		public float spriteScale = 1f;
		/** If True, sprite-based characters will be locked to face a set direction */
		public bool lockDirection = false;
		/** The directional suffix to face (e.g. "L" for "Left"), if lockDirection = True */
		public string spriteDirection = "D";

		private FollowSortingMap followSortingMap;
		private float spriteAngle = 0f;

		/** (DEPRECATED - Use spriteDirectionData instead) */
		public bool doDirections = true;
		/** If True, characters will crossfade between standard animations, if using AnimEngine_SpritesUnity */
		public bool crossfadeAnims = false;
		/** (DEPRECATED - Use spriteDirectionData instead) */
		public bool doDiagonals = false;
		/** The type of frame-flipping to use on sprite-based characters (None, LeftMirrorsRight, RightMirrorsLeft) */
		public AC_2DFrameFlipping frameFlipping = AC_2DFrameFlipping.None;
		/** If True, and frameFlipping != AC_2DFrameFlipping.None, then custom animations will also be flipped */
		public bool flipCustomAnims = false;

		private Vector3 originalScale;
		private bool flipFrames = false;

		/** The direction settings, if using sprite-based characters */
		public SpriteDirectionData _spriteDirectionData = null;

		/** If using Sprites Unity Complex to animate the character, then the 'Body angle float' and 'Head angle float' parameters will be snapped according to this value */
		public AngleSnapping angleSnapping = AngleSnapping.None;

		// Movement variables

		/** The movement speed when walking */
		public float walkSpeedScale = 2f;
		/** The movement speed when running */
		public float runSpeedScale = 6f;
		/** The factor by which speed is reduced when reversing (Tank Controls / First Person only) */
		public float reverseSpeedFactor = 1f;
		/** The turn speed */
		public float turnSpeed = 7f;
		/** The acceleration factor */
		public float acceleration = 6f;
		/** The deceleration factor */
		public float deceleration = 0f;
		/** If True, the character will turn on the spot to face their destination before moving */
		public bool turnBeforeWalking = false;
		/** The minimum distance between the character and its destination for running to be possible */
		public float runDistanceThreshold = 1f;
		/** If True, then sprite-based characters will only move when their sprite frame changes */
		public bool antiGlideMode = false;
		/** Enables 'retro-style' movement when pathfinding, where characters ignore Acceleration and Deceleration values, and turn instantly when moving */
		public bool retroPathfinding = false;

		protected float pathfindUpdateTime = 0f;
		protected bool isJumping = false;

		private float sortingMapScale = 1f;
		private bool isReversing = false;
		protected float turnFloat = 0f;
		private string currentSpriteName = "";
		private SpriteRenderer _spriteRenderer;
		private bool isTurningBeforeWalking;
		private float lastDist;
		private bool isExactLerping;
		private Vector3 newVel;
		private float nonFacingFactor = 1f;
		protected Paths ownPath;

		private Quaternion actualRotation;
		private Vector3 actualForward = Vector3.forward;
		private Vector3 actualRight = Vector3.right;

		// Rigidbody variables

		/** If True, the character will ignore the effects of gravity */
		public bool ignoreGravity = false;
		/** If True, the character's Rigidbody will be frozen in place when idle. This is to help slipping when on sloped surfaces */
		public bool freezeRigidbodyWhenIdle = false;
		/** If True, and the character has a Rigidbody component, then it will be moved by adding forces in FixedUpdate, as opposed to the transform being manipulated in Update */
		public bool useRigidbodyForMovement = true;
		/** If True, and the character has a Rigidbody2D component, then it will be moved by adding forces in FixedUpdate, as opposed to the transform being manipulated in Update */
		public bool useRigidbody2DForMovement = false;

		protected Rigidbody _rigidbody = null;
		protected Rigidbody2D _rigidbody2D = null;
		protected float originalGravityScale = 1f;
		protected Collider _collider = null;
		private CapsuleCollider capsuleCollider;
		private CapsuleCollider[] capsuleColliders;
		protected CharacterController _characterController;

		// Wall detection vargiables

		/** If True, then characters will slow down when walking into walls, if using AnimEngine_Mecanim or AnimEngine_SpritesUnityComplex */
		public bool doWallReduction = false;
		/** The layer that walls are expected to be placed on, if doWallReduction = True */
		public string wallLayer = "Default";
		/** The distance to keep away from walls, if doWallReduction = True */
		public float wallDistance = 0.5f;
		/** If True, and doWallReduction = True, then the wall reduction factor will only affect the Animator move speed float parameter, and not character's actual speed */
		public bool wallReductionOnlyParameter = true;

		private float wallReductionFactor = 1f;
		private Vector3 wallRayOrigin = Vector3.zero;
		private float wallRayForward = 0f;

		// Sound variables

		/** The sound to play when walking */
		public AudioClip walkSound;
		/** The sound to play when running */
		public AudioClip runSound;
		/** The sound to play when the character's speech text is scrolling */
		public AudioClip textScrollClip;
		/** The Sound child to play all non-speech AudioClips on. Speech AudioClips will be placed on the root GameObject */
		public Sound soundChild;

		protected AudioSource audioSource;
		/** The AudioSource from which to play speech audio */
		public AudioSource speechAudioSource;

		protected Paths activePath = null;

		/** If True, the character will run when moving */
		public bool isRunning { get; set; }

		protected float moveSpeed;
		protected Vector3 moveDirection;

		protected int targetNode = 0;
		protected bool pausePath = false;

		protected Vector3 lookDirection;
		private float pausePathTime;
		private ActionList nodeActionList;
		protected int prevNode = 0;

		// Resume path
		private int lastPathPrevNode = 0;
		private int lastPathTargetNode = 0;
		private Paths lastPathActivePath = null;

		protected bool tankTurning = false;

		private Vector2 targetHeadAngles, actualHeadAngles;
		private float headTurnWeight = 0f;

		/** The amount that the head is influenced by IK head-turning. Normally this should be left as 1, but may need reducing if the neck is made of multiple bones. */
		[Range (0f, 1f)] public float headIKTurnFactor = 1f;
		/** The amount that the body is influenced by IK head-turning. */
		[Range (0f, 1f)] public float bodyIKTurnFactor = 0f;
		/** The amount that the eyes are influenced by IK head-turning. */
		[Range (0f, 1f)] public float eyesIKTurnFactor = 1f;

		/** The Transform that the character is turning its head towards, if headFacing != HeadFacing.None */
		public Transform headTurnTarget;
		/** The offset point in World Space that the character is turning its head towards, if headFacing != HeadFacing.None */
		public Vector3 headTurnTargetOffset;
		/** The type of head-turning effect that is currently active (None, Hotspot, Manual) */
		public HeadFacing headFacing = HeadFacing.None;
		/** If True, then inverse-kinematics will be used to turn the character's head dynamically, rather than playing pre-made animations */
		public bool ikHeadTurning = false;
		/** The speed of head-turning */
		public float headTurnSpeed = 4f;

		private Vector3 defaultExactDestination = new Vector3 (0f, 0f, 1234.5f);
		private Sound speechSound;
		
		private bool isUnderTimelineControl;
		private CharacterAnimation2DShot activeCharacterAnimation2DShot;

		protected bool timelineHeadTurnOverride;
		protected Vector3 timelineHeadTurnTargetOffset;
		protected Transform timelineHeadTurnTarget;
		protected float timelineHeadTurnWeight;


		protected void _Awake ()
		{
			if (!CanPhysicallyRotate)
			{
				// Use initial rotation for starting rotation only in this situation
				newRotation = transform.rotation;
				TransformRotation = newRotation;
			}

			newRotation = TransformRotation;

			exactDestination = defaultExactDestination;

			_characterController = GetComponent <CharacterController>();
			if (_characterController)
			{
				wallRayOrigin = _characterController.center;
				wallRayForward = _characterController.radius;
			}
			else
			{
				CapsuleCollider capsuleCollider = GetComponent <CapsuleCollider>();
				if (capsuleCollider)
				{
					wallRayOrigin = capsuleCollider.center;
					wallRayForward = capsuleCollider.radius;
				}
				else
				{
					CircleCollider2D circleCollider = GetComponent <CircleCollider2D>();
					if (circleCollider)
					{
						wallRayOrigin = circleCollider.offset;
						wallRayForward = circleCollider.radius;
					}
					else
					{
						BoxCollider2D boxCollider = GetComponent <BoxCollider2D>();
						if (boxCollider)
						{
							wallRayOrigin = boxCollider.bounds.center;
							wallRayForward = boxCollider.bounds.size.x / 2f;
						}
					}
				}
			}

			#if UNITY_EDITOR
			if (GetComponent <NavMeshAgent>() && GetComponent <NavMeshAgentIntegration>() == null && motionControl != MotionControl.Manual)
			{
				ACDebug.LogWarning ("Charater " + GetName (0) + " has a NavMesh Agent, but no NavMesh Agent Integration component - unless you are using a custom motion controller, add this to make use of the NavMesh Agent.", this);
			}
			#endif

			ownPath = GetComponent<Paths>();
			if (ownPath == null)
			{
				ownPath = gameObject.AddComponent <Paths>();
			}

			if (GetComponentInChildren<FollowSortingMap> ())
			{
				Transform.localScale = Vector3.one;
			}
			originalScale = Transform.localScale;
			charState = CharState.Idle;
			shapeable = GetShapeable ();
			lipSyncTexture = GetComponentInChildren<LipSyncTexture> ();

			speechSound = GetComponent<Sound> ();
			
			ResetAnimationEngine ();
			ResetBaseClips ();

			_animator = GetAnimator ();
			_animation = GetAnimation ();
			SetAntiGlideState ();

			if (spriteChild)
			{
				_spriteRenderer = spriteChild.gameObject.GetComponent<SpriteRenderer> ();
				Vector2 localPosition2D = (Vector2) spriteChild.localPosition;
				if (localPosition2D.magnitude > 0f)
				{
					ACDebug.LogWarning ("The sprite child of '" + gameObject.name + "' is not positioned at (0,0,0) - is this correct?", gameObject);
				}
			}

			if (spriteChild)
			{
				followSortingMap = spriteChild.GetComponent <FollowSortingMap>();
			}
			if (followSortingMap == null)
			{
				followSortingMap = GetComponentInChildren <FollowSortingMap>();
			}

			if (speechAudioSource == null)
			{
				speechAudioSource = GetComponent<AudioSource> ();
			}

			if (soundChild && audioSource == null)
			{
				audioSource = soundChild.GetComponent<AudioSource> ();
			}

			if (_rigidbody == null)
			{
				_rigidbody = GetComponent<Rigidbody> ();
			}

			if (_rigidbody == null)
			{
				_rigidbody2D = GetComponent<Rigidbody2D> ();
				if (_rigidbody2D)
				{
					originalGravityScale = _rigidbody2D.gravityScale;
					if (originalGravityScale <= 0f) originalGravityScale = 1f;

					if (SceneSettings.CameraPerspective != CameraPerspective.TwoD)
					{
						ACDebug.LogWarning ("In order to move a sprite-based character (" + gameObject.name + ") in 3D, there must not be a Rigidbody2D component on the base.", gameObject);
					}
					else if (antiGlideMode)
					{
						_rigidbody2D.isKinematic = true;
						_rigidbody2D = null;
						ACDebug.LogWarning ("The use of character " + gameObject.name + "'s Rigidbody2D component is disabled as it conflicts with the 'Only move when sprite changes feature.", gameObject);
					}
				}
			}
			PhysicsUpdate ();

			if (_collider == null)
			{
				_collider = GetComponent<Collider> ();
				if (_collider && _collider is CapsuleCollider)
				{
					capsuleCollider = _collider as CapsuleCollider;
				}
			}
			capsuleColliders = GetComponentsInChildren<CapsuleCollider> ();

			AdvGame.AssignMixerGroup (speechAudioSource, SoundType.Speech);
			AdvGame.AssignMixerGroup (audioSource, SoundType.SFX);

			displayLineID = lineID;

			// Initialise
			if (GetAnimation ())
			{
				// Hack: Force idle of Legacy characters
				AdvGame.PlayAnimClip (GetAnimation (), AdvGame.GetAnimLayerInt (AnimLayer.Base), idleAnim, AnimationBlendMode.Blend, WrapMode.Loop, 0f, null, false);
			}
			else if (spriteChild)
			{
				// Hack: update 2D sprites
				InitSpriteChild ();
			}
			UpdateScale ();

			#if UNITY_EDITOR
			if (GetAnimator () && customAnimator == null)
			{
				int numAnimators = GetComponentsInChildren<Animator>().Length;
				if (numAnimators > 1)
				{
					ACDebug.Log ("Multiple Animator components detected on the character '" + name + "' - the one on object '" + GetAnimator ().name + "' will be used.", GetAnimator ());
				}
			}
			#endif

			GetAnimEngine ().TurnHead (Vector2.zero);
			GetAnimEngine ().PlayIdle ();
		}


		protected virtual void OnEnable ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Register (this);
			//EventManager.OnInitialiseScene += OnInitialiseScene;
			EventManager.OnManuallyTurnACOff += OnManuallyTurnACOff;
			EventManager.OnStartSpeech_Alt += OnStartSpeech;
			EventManager.OnStopSpeech_Alt += OnStopSpeech;
		}


		private void Start ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Register (this);
		}


		protected virtual void OnDisable ()
		{
			if (KickStarter.stateHandler) KickStarter.stateHandler.Unregister (this);
			//EventManager.OnInitialiseScene -= OnInitialiseScene;
			EventManager.OnManuallyTurnACOff -= OnManuallyTurnACOff;
			EventManager.OnStartSpeech_Alt -= OnStartSpeech;
			EventManager.OnStopSpeech_Alt -= OnStopSpeech;
		}


		/** Returns True if the character is a Player, even if they are not the current active Player. */
		public virtual bool IsPlayer
		{
			get
			{
				return false;
			}
		}


		/** The character's "Update" function, called by StateHandler. */
		public virtual void _Update ()
		{
			BaseUpdate ();
		}


		protected void BaseUpdate ()
		{
			CalculateHeadTurn ();
			UpdateWallReductionFactor ();
			CalcHeightChange ();

			if (spriteChild && KickStarter.settingsManager)
			{
				PrepareSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
			}

			AnimUpdate ();
			SpeedUpdate ();

			PathUpdate ();
			PhysicsUpdate ();

			if (!antiGlideMode)
			{
				MoveUpdate ();
			}
		}


		/** The character's "LateUpdate" function, called by StateHandler. */
		public void _LateUpdate ()
		{
			if (antiGlideMode)
			{
				MoveUpdate ();

				if (spriteChild && KickStarter.settingsManager)
				{
					PrepareSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
				}
			}

			if (spriteChild && KickStarter.settingsManager)
			{
				UpdateSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
			}
			UpdateScale ();
		}


		/**
		 * The character's "FixedUpdate" function, called by StateHandler.
		 */
		public virtual void _FixedUpdate ()
		{
			MoveRigidbody ();
			DoTurn (true);
		}


		private void OnAnimatorIK (int layerIndex)
		{
			if (animEngine.IKEnabled)
			{
				if (timelineHeadTurnOverride)
				{
					_animator.SetLookAtPosition (timelineHeadTurnTarget.position + timelineHeadTurnTarget.TransformVector (timelineHeadTurnTargetOffset));
					_animator.SetLookAtWeight (timelineHeadTurnWeight, bodyIKTurnFactor, headIKTurnFactor, eyesIKTurnFactor);
				}
				else if (ikHeadTurning)
				{
					if (headTurnWeight > 0f && headTurnTarget)
					{
						Vector3 position = CalculateIKHeadTurnPosition ();
						_animator.SetLookAtPosition (position);
					}
					_animator.SetLookAtWeight (headTurnWeight, bodyIKTurnFactor, headIKTurnFactor, eyesIKTurnFactor);
				}

				leftHandIKController.OnAnimatorIK (_animator, AvatarIKGoal.LeftHand);
				rightHandIKController.OnAnimatorIK (_animator, AvatarIKGoal.RightHand);
			}
		}


		private Vector3 CalculateIKHeadTurnPosition ()
		{
			Vector3 headPosition = Transform.position;
			if (neckBone)
			{
				headPosition += (neckBone.position - Transform.position);
			}
			else if (_characterController)
			{
				headPosition += new Vector3 (0f, _characterController.height * Transform.localScale.y * 0.8f, 0f);
			}
			else if (capsuleCollider)
			{
				headPosition += new Vector3 (0f, capsuleCollider.height * Transform.localScale.y * 0.8f, 0f);
			}

			Vector3 rightDirection = Vector3.RotateTowards (TransformForward, TransformRight, actualHeadAngles.x * Mathf.Deg2Rad, 1f);
			return headPosition + rightDirection + (Vector3.up * Mathf.Asin (actualHeadAngles.y * Mathf.Deg2Rad));
		}


		#if UNITY_EDITOR

		private void OnDrawGizmos ()
		{
			if (!retroPathfinding && KickStarter.settingsManager)
			{
				Gizmos.color = Color.yellow;
				Gizmos.DrawWireSphere (Transform.position, KickStarter.settingsManager.GetDestinationThreshold ());
			}

			if (ikHeadTurning && headTurnTarget)
			{
				Gizmos.color = Color.red;
				Gizmos.DrawWireSphere (CalculateIKHeadTurnPosition (), 0.1f);
			}
		}

		#endif


		/**
		 * <summary>Rebuilds the active pathfind.</summary>
		 */
		public void RecalculateActivePathfind ()
		{
			// Caveat: If pathfindUpdateTime < 0, the character is not pathfinding and just heading in a straight line (special case), so don't recalculate
			if (activePath && !pausePath && activePath == ownPath && pathfindUpdateTime >= 0f)
			{
				Vector3 targetPosition = activePath.nodes[activePath.nodes.Count - 1];
				MoveToPoint (targetPosition, isRunning, true);
			}
		}


		protected void PathUpdate ()
		{
			if (activePath && activePath.nodes.Count > 0)
			{
				if (pausePath)
				{
					if (nodeActionList)
					{
						if (!KickStarter.actionListManager.IsListRunning (nodeActionList))
						{
							SetNextNodes (true);
						}
					}
					else if (Time.time > pausePathTime)
					{
						SetNextNodes (true);
					}
					return;
				}
				else
				{
					if (targetNode == -1 || targetNode >= activePath.nodes.Count)
					{
						ACDebug.LogWarning ("Invalid node target - cannot update pathfinding on " + name, this);
						return;
					}
					if (pathfindUpdateTime > 0f)
					{
						pathfindUpdateTime -= Time.deltaTime;
						if (pathfindUpdateTime <= 0f)
						{
							pathfindUpdateTime = 0f;
							RecalculateActivePathfind ();
						}
					}

					if (targetNode >= activePath.nodes.Count) return;

					Vector3 direction = activePath.nodes[targetNode] - Transform.position;
					Vector3 lookDir = new Vector3 (direction.x, 0f, direction.z);

					if (SceneSettings.IsUnity2D ())
					{
						direction.z = 0f;
						SetMoveDirection (direction);

						lookDir = new Vector3 (direction.x, 0f, direction.y);

						SetLookDirection (lookDir, false);
					}
					else if (activePath.affectY)
					{
						SetMoveDirection (direction);
						SetLookDirection (lookDir, false);
					}
					else
					{
						SetLookDirection (lookDir, false);
						SetMoveDirectionAsForward ();
					}

					if (isRunning && direction.magnitude > 0 && direction.magnitude < runDistanceThreshold)
					{
						if (WillStopAtNextNode ())
						{
							isRunning = false;
						}
					}
					
					float nodeThreshold = KickStarter.settingsManager.GetDestinationThreshold ();
					if (isRunning && GetMotionControl () == MotionControl.Automatic)
					{
						float multiplier = (1 - (runSpeedScale / walkSpeedScale)) / 20f;
						multiplier *= Mathf.Clamp (GetDeceleration (), 1f, 20f);
						multiplier += (runSpeedScale / walkSpeedScale);

						nodeThreshold *= multiplier;
					}

					if (retroPathfinding)
					{
						nodeThreshold = 0.01f;
					}

					float directionMagnitude = direction.magnitude;
					if ((SceneSettings.IsUnity2D () && directionMagnitude < nodeThreshold) ||
						(activePath.affectY && directionMagnitude < nodeThreshold) ||
						(!activePath.affectY && lookDir.magnitude < nodeThreshold))
					{
						KickStarter.eventManager.Call_OnCharacterReachNode (this, activePath, targetNode);
						if (activePath.nodeCommands.Count > targetNode)
						{
							NodeCommand nodeCommand = activePath.nodeCommands[targetNode];
							nodeCommand.SetParameter (activePath.commandSource, this.gameObject);

							if (activePath.commandSource == ActionListSource.InScene && nodeCommand.cutscene && nodeCommand.pausesCharacter)
							{
								PausePath (activePath.nodePause, nodeCommand.cutscene);
							}
							else if (activePath.commandSource == ActionListSource.AssetFile && nodeCommand.actionListAsset && nodeCommand.pausesCharacter)
							{
								PausePath (activePath.nodePause, nodeCommand.actionListAsset);
							}
							else if (activePath.nodePause > 0f)
							{
								PausePath (activePath.nodePause);
							}
							else
							{
								SetNextNodes ();
							}
						}
						else if (activePath.nodePause > 0f)
						{
							PausePath (activePath.nodePause);
						}
						else
						{
							SetNextNodes ();
						}
						return;
					}
				}
			}
		}


		private void SpeedUpdate ()
		{
			if (AnimationControlledByAnimationShot)
			{
				moveSpeed = moveSpeedLerp.Update (moveSpeed, GetTargetSpeed (), acceleration);
			}
			else if (charState == CharState.Move)
			{
				lastDist = Mathf.Infinity;
				Accelerate ();
			}
			else
			{
				if (charState == CharState.Decelerate || charState == CharState.Custom)
				{
					Decelerate ();
				}
				else if (charState == CharState.Idle)
				{
					if (moveSpeed > 0f)
					{
						lastDist = Mathf.Infinity;
						moveSpeed = 0f;
					}
					isExactLerping = false;
				}
			}
		}


		private void PhysicsUpdate ()
		{
			if (GetMotionControl () == MotionControl.Manual || GetMotionControl () == MotionControl.JustTurning)
			{
				return;
			}

			if (_rigidbody)
			{
				if (ignoreGravity)
				{
					_rigidbody.useGravity = false;
				}
				else if (charState == CharState.Custom && moveSpeed < 0.01f)
				{
					_rigidbody.useGravity = false;
				}
				else
				{
					if (activePath && activePath.affectY)
					{
						_rigidbody.useGravity = false;
					}
					else
					{
						_rigidbody.useGravity = true;
					}
				}
			}
			else if (_rigidbody2D)
			{
				if (ignoreGravity)
				{
					_rigidbody2D.gravityScale = 0f;
				}
				else if (charState == CharState.Custom && moveSpeed < 0.01f)
				{
					_rigidbody2D.gravityScale = 0f;
				}
				else
				{
					if (activePath && activePath.affectY)
					{
						_rigidbody2D.gravityScale = 0f;
					}
					else
					{
						_rigidbody2D.gravityScale = originalGravityScale;
					}
				}
			}
		}


		private void AnimUpdate ()
		{
			if (animEngine == null) return;

			if (isTalking)
			{
				ProcessLipSync ();
			}

			if (IsMovingHead () || animEngine.updateHeadAlways)
			{
				AnimateHeadTurn ();
			}

			if (AnimationControlledByAnimationShot)
			{
				return;
			}
			
			if (isJumping)
			{
				animEngine.PlayJump ();
				StopStandardAudio ();
			}
			else
			{
				switch (charState)
				{
					case CharState.Idle:
					case CharState.Decelerate:
						if (IsTurning ())
						{
							if (turnFloat < 0f)
							{
								animEngine.PlayTurnLeft ();
							}
							else
							{
								animEngine.PlayTurnRight ();
							}
						}
						else
						{
							if (isTalking && (talkingAnimation == TalkingAnimation.Standard || animationEngine == AnimationEngine.Custom))
							{
								animEngine.PlayTalk ();
							}
							else
							{
								animEngine.PlayIdle ();
							}
						}

						StopStandardAudio ();
						break;

					case CharState.Move:
						if (isRunning)
						{
							animEngine.PlayRun ();
						}
						else
						{
							animEngine.PlayWalk ();
						}

						PlayStandardAudio ();
						break;

					default:
						StopStandardAudio ();
						break;
				}
			}

			animEngine.PlayVertical ();

			if (animEngine.IKEnabled)
			{
				leftHandIKController.Update ();
				rightHandIKController.Update ();
			}
		}


		private void MoveUpdate ()
		{
			if (animEngine)
			{
				if (GetMotionControl () == MotionControl.Automatic)
				{
					RootMotionType rootMotionType = GetRootMotionType ();

					if (Mathf.Approximately (moveSpeed, 0f))
					{
						newVel = Vector3.zero;
					}

					if (moveSpeed > 0f && rootMotionType != RootMotionType.ThreeD)
					{
						newVel = moveDirection * moveSpeed * walkSpeedScale * sortingMapScale;
						if (isReversing)
						{
							newVel *= reverseSpeedFactor;
						}

						if (SceneSettings.IsTopDown ())
						{
							float magnitude = newVel.magnitude;

							if (magnitude > 0f)
							{
								float upAmount = Mathf.Abs (Vector3.Dot (newVel.normalized, Vector3.forward));
								float mag = (magnitude * (1f - upAmount)) + (magnitude * KickStarter.sceneSettings.GetVerticalReductionFactor () * upAmount);
								newVel *= mag / magnitude;
							}
						}
						else if (SceneSettings.IsUnity2D ())
						{
							newVel.z = 0f;
							float magnitude = newVel.magnitude;

							if (magnitude > 0f)
							{
								float upAmount = Mathf.Abs (Vector3.Dot (newVel.normalized, Vector3.up));
								float mag = (magnitude * (1f - upAmount)) + (magnitude * KickStarter.sceneSettings.GetVerticalReductionFactor () * upAmount);
								newVel *= mag / magnitude;
							}
						}
						else
						{
							newVel *= GetNonFacingReductionFactor ();
							newVel *= (doWallReduction && !wallReductionOnlyParameter) ? wallReductionFactor : 1f;
						}

						newVel *= KickStarter.playerInput.GetDragMovementSlowDown ();

						bool noMove = false;
						if (rootMotionType == RootMotionType.TwoD)
						{
							if (spriteDirection == "L" || spriteDirection == "R")
							{
								newVel.x = 0f;
							}
							else if (spriteDirection == "U" || spriteDirection == "D")
							{
								newVel.y = 0f;
							}
							else
							{
								newVel = Vector3.zero;
							}
						}
						else if (antiGlideMode)
						{
							if (_spriteRenderer && _spriteRenderer.sprite)
							{
								string newSpriteName = _spriteRenderer.sprite.name;
								if (newSpriteName == currentSpriteName)
								{
									noMove = true;
								}
								else
								{
									currentSpriteName = newSpriteName;
								}
							}
						}

						if (!noMove && !AnimationControlledByAnimationShot)
						{
							float _deltaTime = (antiGlideMode) ? Time.fixedDeltaTime : Time.deltaTime;

							if (_deltaTime > 0f)
							{
								if (DoRigidbodyMovement () || DoRigidbodyMovement2D ())
								{ }
								else if (_characterController)
								{
									if (!ignoreGravity)
									{
										if (IsGrounded ())
										{
											newVel.y = -_characterController.stepOffset / Time.deltaTime;
										}
										else
										{
											newVel += Physics.gravity;
										}
									}

									_characterController.Move (newVel * _deltaTime);
								}
								else if (retroPathfinding && IsMovingAlongPath ())
								{
									float frameSpeed = ((isRunning) ? runSpeedScale : walkSpeedScale) * sortingMapScale;

									float upAmount = Mathf.Abs (Vector2.Dot ((GetTargetPosition () - Transform.position).normalized, Vector2.up));
									float mag = (1f - upAmount) + (KickStarter.sceneSettings.GetVerticalReductionFactor () * upAmount);
									frameSpeed *= mag;

									Vector3 newPosition = Vector3.MoveTowards (Transform.position, GetTargetPosition (), frameSpeed * _deltaTime);
									Transform.position = newPosition;
								}
								else
								{
									Transform.position += (newVel * _deltaTime);
								}

								if (antiGlideMode && KickStarter.mainCamera && KickStarter.mainCamera.attachedCamera)
								{
									// Update camera position again if we're moving this frame in anti-glide mode
									KickStarter.mainCamera.attachedCamera._Update ();
								}
							}
						}
					}
					else
					{
						if (_characterController && rootMotionType != RootMotionType.ThreeD)
						{
							if (!_characterController.isGrounded && !ignoreGravity)
							{
								_characterController.Move (Physics.gravity * Time.deltaTime);
							}
						}
					}
				}
			}

			Turn (false);
			DoTurn ();
		}


		private Vector3 CalcForce (Vector3 targetVelocity, float verticalSpeed, float mass, Vector3 rigidbodyVelocity)
		{
			targetVelocity.y += verticalSpeed;

			Vector3 deltaVelocity = targetVelocity - rigidbodyVelocity;
			Vector3 force = mass * deltaVelocity / Time.deltaTime * Time.fixedDeltaTime / 0.02f;

			return force;
		}


		private void MoveRigidbody ()
		{
			if (GetMotionControl () == MotionControl.Manual || GetMotionControl () == MotionControl.JustTurning)
			{
				return;
			}

			if (DoRigidbodyMovement ())
			{
				if (GetRootMotionType () == RootMotionType.None)
				{
					if (_rigidbody.isKinematic)
					{
						_rigidbody.MovePosition (Transform.position + newVel * Time.deltaTime);
					}
					else
					{
						Vector3 force = CalcForce (newVel, _rigidbody.velocity.y, _rigidbody.mass, _rigidbody.velocity);
						_rigidbody.AddForce (force);
					}
				}

				bool xFrozen = false;
				bool yFrozen = false;
				bool zFrozen = false;

				if (freezeRigidbodyWhenIdle)
				{
					xFrozen = yFrozen = zFrozen = (!isJumping && (charState == CharState.Custom || charState == CharState.Idle));
				}
				else
				{
					xFrozen = (_rigidbody.constraints & RigidbodyConstraints.FreezePositionX) != RigidbodyConstraints.None;
					yFrozen = (_rigidbody.constraints & RigidbodyConstraints.FreezePositionY) != RigidbodyConstraints.None;
					zFrozen = (_rigidbody.constraints & RigidbodyConstraints.FreezePositionZ) != RigidbodyConstraints.None;
				}

				_rigidbody.constraints = RigidbodyConstraints.FreezeRotationX |
										 RigidbodyConstraints.FreezeRotationY |
										 RigidbodyConstraints.FreezeRotationZ |
										 ((xFrozen) ? RigidbodyConstraints.FreezePositionX : 0) |
										 ((yFrozen) ? RigidbodyConstraints.FreezePositionY : 0) |
										 ((zFrozen) ? RigidbodyConstraints.FreezePositionZ : 0);
			}
			else if (DoRigidbodyMovement2D ())
			{
				if (GetRootMotionType () == RootMotionType.None)
				{
					if (_rigidbody2D.isKinematic)
					{
						_rigidbody2D.MovePosition (Transform.position + newVel * Time.deltaTime);
					}
					else
					{
						Vector3 force = CalcForce (newVel, 0f, _rigidbody2D.mass, _rigidbody2D.velocity);
						_rigidbody2D.AddForce (force);
					}
				}


				bool xFrozen = false;
				bool yFrozen = false;

				if (freezeRigidbodyWhenIdle)
				{
					xFrozen = yFrozen = (!isJumping && (charState == CharState.Custom || charState == CharState.Idle));
				}
				else
				{
					xFrozen = (_rigidbody2D.constraints & RigidbodyConstraints2D.FreezePositionX) != RigidbodyConstraints2D.None;
					yFrozen = (_rigidbody2D.constraints & RigidbodyConstraints2D.FreezePositionY) != RigidbodyConstraints2D.None;
				}

				_rigidbody2D.constraints = RigidbodyConstraints2D.FreezeRotation |
										   ((xFrozen) ? RigidbodyConstraints2D.FreezePositionX : 0) |
										   ((yFrozen) ? RigidbodyConstraints2D.FreezePositionY : 0);

			}
		}


		protected float GetTargetSpeed ()
		{
			if (animatorIsOnRoot)
			{
				/**
				 * Stupid bug alert!
				 *
				 * A check was made way back in v1.28 for a root Animator component because back then only Legacy and Mecanim animation was available (no 2D!).
				 * With this animation mode, the "move speed float" had to be tied directly to the walk/run speeds so that transitions could be properly built.
				 * This wasn't the right way to do this, of course, but the problem is now everyone's using this - so it can't be changed without having to get basically
				 * everyone working in 3D to update their movement values and transitions.
				 *
				 * Since you can still achieve the same movement speeds as non-animator-on-root characters by lowering your movement values,
				 * all things considered it's probably best to just leave as is and say 'Yes I was stupid back then'! :/
				 */

				if (isRunning)
				{
					return runSpeedScale;
				}
				else
				{
					return walkSpeedScale;
				}
			}
			else
			{
				if (AnimationControlledByAnimationShot)
				{
					if (isRunning)
					{
						return runSpeedScale / walkSpeedScale;
					}
					return 1f;
				}

				if (isRunning)
				{
					return moveDirection.magnitude * runSpeedScale / walkSpeedScale;
				}
				return moveDirection.magnitude;
			}
		}


		private Vector3 GetSmartPosition (Vector3 targetPoint)
		{
			if (SceneSettings.IsUnity2D ())
			{
				targetPoint.z = Transform.position.z;
			}
			else
			{
				targetPoint.y = Transform.position.y;
			}
			return targetPoint;
		}


		protected void AccurateAcc (float targetSpeed, bool canStop)
		{
			if (IsTurningBeforeWalking ())
			{
				return;
			}

			float lerpDistance = 3f * ((isRunning) ? (runSpeedScale / walkSpeedScale) : 1f) / GetDeceleration ();
			float dist = Vector3.Distance (GetSmartPosition (exactDestination), Transform.position);

			if (canStop && dist <= 0f)
			{
				charState = CharState.Idle;
				Transform.position = GetSmartPosition (exactDestination);
				moveSpeed = 0f;
				isExactLerping = false;
				exactDestination = defaultExactDestination;
				return;
			}
			else if (canStop && (dist < lerpDistance / 4f || dist > lastDist))
			{
				isExactLerping = true;
				Transform.position = exactPositionLerp.Update (Transform.position, GetSmartPosition (exactDestination), GetDeceleration ());
			}
			else
			{
				isExactLerping = false;
			}

			if (dist < lastDist)
			{
				float fac = 1f;
				if (dist < lerpDistance)
				{
					isRunning = false;
					fac = dist / lerpDistance;
				}

				moveSpeed = moveSpeedLerp.Update (moveSpeed, targetSpeed * fac, (canStop) ? GetDeceleration () : acceleration);
			}

			if (canStop)
			{
				moveDirection = GetSmartPosition (exactDestination) - Transform.position;
				lastDist = dist;
			}
		}


		protected virtual void Accelerate ()
		{
			if (AccurateDestination () && WillStopAtNextNode ())
			{
				AccurateAcc (GetTargetSpeed (), false);
			}
			else
			{
				moveSpeed = moveSpeedLerp.Update (moveSpeed, GetTargetSpeed (), acceleration);
			}
		}


		private void Decelerate ()
		{
			if (animEngine != null && animEngine.turningStyle == TurningStyle.Linear)
			{
				moveSpeed = moveSpeedLerp.Update (moveSpeed, 0f, GetDeceleration () * 3f);
			}
			else
			{
				if (AccurateDestination () && !CanBeDirectControlled () && charState == CharState.Decelerate)
				{
					AccurateAcc (moveSpeed, true);
				}
				else
				{
					exactDestination = defaultExactDestination;
					moveSpeed = moveSpeedLerp.Update (moveSpeed, 0f, GetDeceleration ());
				}
			}

			if (moveSpeed <= 0f)
			{
				moveSpeed = 0f;

				if (charState != CharState.Custom)
				{
					charState = CharState.Idle;
				}
			}
		}


		/**
		 * <summary>Checks if the character should attempt to be as accurate as possible when moving to a destination.</summary>
		 * <returns>True if the character should attempt to be as accurate as possible when moving to a destination.</returns>
		 */
		public bool AccurateDestination ()
		{
			if (retroPathfinding)
			{
				return false;
			}

			if (motionControl == MotionControl.Automatic &&
				KickStarter.settingsManager.experimentalAccuracy &&
				KickStarter.settingsManager.destinationAccuracy >= 1f &&
				(!this.IsPlayer || KickStarter.settingsManager.movementMethod != MovementMethod.StraightToCursor) &&
				exactDestination != defaultExactDestination)
			{
				return true;
			}
			return false;
		}


		private float GetDeceleration ()
		{
			if (deceleration <= 0f)
			{
				return acceleration;
			}
			return deceleration;
		}


		/**
		 * <summary>Teleports the character. The message 'OnTeleport' will be sent to the character's GameObject after the position is set.</summary>
		 * <param name = "_position">The point to teleport to</param>
		 * <param name = "recalculateActivePathFind">If True and the character is walking, then their navigation path will be recalculated to account for the position change</param>
		 */
		public void Teleport (Vector3 _position, bool recalculateActivePathFind = false)
		{
			bool enableCharacterController = false;
			if (_characterController)
			{
				enableCharacterController = _characterController.enabled;
				_characterController.enabled = false;
			}

			Transform.position = _position;

			if (enableCharacterController)
			{
				_characterController.enabled = true;
			}

			if (recalculateActivePathFind)
			{
				RecalculateActivePathfind ();
			}

			prevHeight = _position.y;

			SendMessage ("OnTeleport", SendMessageOptions.DontRequireReceiver);
		}


		/**
		 * <summary>Instantly rotates the character</summary>
		 * <param name = "_rotation">The new rotation</param>
		 */
		public void SetRotation (Quaternion _rotation)
		{
			TransformRotation = _rotation;
			SetLookDirection (TransformForward, true);
		}


		/**
		 * <summary>Instantly rotates the Player</summary>
		 * <param name = "angle">The angle, in degrees, around the Y-axis to rotate to</param>
		 */
		public void SetRotation (float angle)
		{
			TransformRotation = Quaternion.AngleAxis (angle, Vector3.up);
			SetLookDirection (TransformForward, true);
		}


		/**
		 * Instantly stops the character turning.
		 */
		public void StopTurning ()
		{
			SetLookDirection (TransformForward, true);
			newRotation = TransformRotation;

			StopTankTurning ();

			newAngleLerp.Reset ();
			newAngleLinearLerp.Reset ();
		}


		public virtual void StopTankTurning ()
		{ }


		private void Turn (bool isInstant)
		{
			if (lookDirection == Vector3.zero)
			{
				return;
			}

			if (turnSpeed < 0f)
			{
				isInstant = true;
			}

			if (retroPathfinding && IsMovingAlongPath () && !IsTurningBeforeWalking ())
			{
				isInstant = true;
			}

			if (isInstant)
			{
				turnFloat = 0f;
				Quaternion targetRotation = GetTargetRotation ();
				if (motionControl != MotionControl.Manual)
				{
					TransformRotation = targetRotation;
				}
				newRotation = targetRotation;

				if (KickStarter.settingsManager && spriteChild)
				{
					PrepareSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
				}

				SendMessage ("OnSnapRotate", SendMessageOptions.DontRequireReceiver);
				return;
			}

			float targetAngle = Mathf.Atan2 (lookDirection.x, lookDirection.z);
			float currentAngle = Mathf.Atan2 (TransformForward.x, TransformForward.z);

			float angleDiff = targetAngle - currentAngle;

			if (Mathf.Approximately (angleDiff, 0f))
			{
				turnFloat = turnFloatLerp.Update (turnFloat, 0f, turnSpeed);
				newRotation = TransformRotation;
				return;
			}
			else if (angleDiff < -Mathf.PI)
			{
				targetAngle += Mathf.PI * 2f;
				angleDiff += Mathf.PI * 2f;
			}
			else if (angleDiff > Mathf.PI)
			{
				targetAngle -= Mathf.PI * 2f;
				angleDiff -= Mathf.PI * 2f;
			}

			if (retroPathfinding && IsTurningBeforeWalking () && SceneSettings.IsUnity2D ())
			{
				if (targetAngle * currentAngle < 0f && Mathf.Abs (angleDiff) > (Mathf.PI / 2f))
				{
					// Retro mode: Turn facing camera if > 45 degrees
					if (currentAngle < 0f)
					{
						targetAngle -= Mathf.PI * 2f;
						angleDiff -= Mathf.PI * 2f;
					}
					else
					{
						targetAngle += Mathf.PI * 2f;
						angleDiff += Mathf.PI * 2f;
					}
				}
			}

			float turnMultiplier = 0f;
			// Min prevents turn direction flipping when slowing down
			if (angleDiff > 0f)
			{
				turnMultiplier = Mathf.Min (angleDiff / 2f, 1f);
			}
			else if (angleDiff < 0f)
			{
				turnMultiplier = Mathf.Max (angleDiff / 2f, -1f);
			}
			turnFloat = turnFloatLerp.Update (turnFloat, turnSpeed * turnMultiplier, turnSpeed);

			if (animEngine && animEngine.turningStyle == TurningStyle.Linear)
			{
				float newAngle = newAngleLinearLerp.Update (currentAngle, targetAngle, turnSpeed * GetScriptTurningFactor ());
				newRotation = Quaternion.AngleAxis (newAngle * Mathf.Rad2Deg, Vector3.up);
			}
			else
			{
				float newAngle = newAngleLerp.Update (currentAngle, targetAngle, turnSpeed * GetScriptTurningFactor ());
				newRotation = Quaternion.AngleAxis (newAngle * Mathf.Rad2Deg, Vector3.up);
			}
		}


		/**
		 * <summary>Gets the difference between the character's current facing angle, and the angle they wish to face.</summary>
		 * <returns>The difference between the character's current facing angle, and the angle they wish to face.</summary>
		 */
		public float GetAngleDifference ()
		{
			float targetAngle = Mathf.Atan2 (lookDirection.x, lookDirection.z);
			float currentAngle = Mathf.Atan2 (TransformForward.x, TransformForward.z);

			return (targetAngle - currentAngle);
		}


		private float GetNonFacingReductionFactor ()
		{
			if (activePath)
			{
				Vector3 idealDir = Vector3.zero;
				if (SceneSettings.IsUnity2D ())
				{
					Vector2 idealDir2D = GetTargetPosition () - Transform.position;
					idealDir = new Vector3 (idealDir2D.x, 0f, idealDir2D.y);
				}
				else
				{
					idealDir = GetSmartPosition (GetTargetPosition ()) - Transform.position;
				}
				float scale = Vector3.Dot (TransformForward, idealDir.normalized);
				scale = (scale * scale);
				return Mathf.Clamp01 (scale);
			}
			else
			{
				return 1f;
			}
		}


		private void UpdateWallReductionFactor ()
		{
			if (!doWallReduction) return;
			
			if (SceneSettings.CameraPerspective == CameraPerspective.TwoD)
			{
				// 2D

				Vector2 forwardVector = TransformForward;
				if (SceneSettings.IsUnity2D ())
				{
					forwardVector = new Vector2 (TransformForward.x, TransformForward.z);
				}

				Vector2 origin = (Vector2) Transform.position + (Vector2) wallRayOrigin + (forwardVector * wallRayForward);
				RaycastHit2D hit = UnityVersionHandler.Perform2DRaycast (origin, forwardVector, wallDistance, 1 << LayerMask.NameToLayer (wallLayer));

				if (hit.collider)
				{
					wallReductionFactor = wallReductionLerp.Update (wallReductionFactor, (hit.point - origin).magnitude / wallDistance, 10f);
				}
				else
				{
					wallReductionFactor = wallReductionLerp.Update (wallReductionFactor, 1f, 10f);
				}
			}
			else
			{
				// 3D

				Vector3 origin = Transform.position + wallRayOrigin + (TransformForward * wallRayForward);
				RaycastHit hit;
				if (Physics.Raycast (origin, TransformForward, out hit, wallDistance, 1 << LayerMask.NameToLayer (wallLayer)))
				{
					wallReductionFactor = wallReductionLerp.Update (wallReductionFactor, (hit.point - origin).magnitude / wallDistance, 10f);
				}
				else
				{
					wallReductionFactor = wallReductionLerp.Update (wallReductionFactor, 1f, 10f);
				}
			}
		}


		private float GetScriptTurningFactor ()
		{
			if (_animator && _animator.applyRootMotion)
			{
				return (1f - rootTurningFactor);
			}
			return 1f;
		}


		private void DoTurn (bool fromFixedUpdate = false)
		{
			if (GetMotionControl () == MotionControl.Manual || lookDirection == Vector3.zero)
			{
				return;
			}

			if (fromFixedUpdate == DoRigidbodyMovement ())
			{
				if (GetScriptTurningFactor () <= 0f)
				{
					return;
				}

				if (fromFixedUpdate && GetScriptTurningFactor () >= 1f)
				{
					_rigidbody.MoveRotation (newRotation);
				}
				else
				{
					TransformRotation = newRotation;
				}
			}
		}


		/**
		 * <summary>Sets the intended direction to face.</summary>
		 * <param name = "_direction">The relative direction to face</param>
		 * <param name = "isInstant">If True, the Player will instantly turn to face the direction</param>
		 */
		public void SetLookDirection (Vector3 _direction, bool isInstant)
		{
			Vector3 oldLookDirection = lookDirection;
			lookDirection = new Vector3 (_direction.x, 0f, _direction.z);

			if (KickStarter.settingsManager.rotationsAffectedByVerticalReduction && SceneSettings.IsUnity2D () && KickStarter.settingsManager.movementMethod == MovementMethod.PointAndClick)
			{
				if (_direction != TransformForward)
				{
					// Modify 3D direction so that it appears more correct in 2D
					lookDirection.z /= KickStarter.sceneSettings.GetVerticalReductionFactor ();
				}
			}

			if (isInstant)
			{
				Turn (isInstant);
			}

			if (oldLookDirection.normalized != lookDirection.normalized)
			{
				KickStarter.eventManager.Call_OnSetLookDirection (this, lookDirection, isInstant);
			}
		}


		/**
		 * <summary>Moves the character in a particular direction.</summary>
		 * <param name = "_direction">The direction to move in</param>
		 * <param name = "useSmoothing">If True, smoothing will be applied to the change in direction, so the change will be gradual</param>
		 */
		public void SetMoveDirection (Vector3 _direction, bool useSmoothing = false)
		{
			if (_direction != Vector3.zero)
			{
				Quaternion targetRotation = Quaternion.LookRotation (_direction, Vector3.up);

				Vector3 newMoveDirection = targetRotation * Vector3.forward;
				newMoveDirection.Normalize ();

				if (useSmoothing)
				{
					moveDirection = Vector3.Lerp (moveDirection, newMoveDirection, Time.deltaTime * 12f);
				}
				else
				{
					moveDirection = newMoveDirection;
				}
			}
		}


		/** Moves the character forward. */
		public void SetMoveDirectionAsForward ()
		{
			isReversing = false;
			moveDirection = TransformForward;
			if (SceneSettings.IsUnity2D ())
			{
				moveDirection = new Vector3 (moveDirection.x, moveDirection.z, 0f);
			}
			moveDirection.Normalize ();
		}


		/** Moves the character backward. */
		public void SetMoveDirectionAsBackward ()
		{
			isReversing = true;
			moveDirection = -TransformForward;
			if (SceneSettings.IsUnity2D ())
			{
				moveDirection = new Vector3 (moveDirection.x, moveDirection.z, 0f);
			}
			moveDirection.Normalize ();
		}


		private void SetAntiGlideState ()
		{
			if (!animEngine.isSpriteBased || GetRootMotionType () != RootMotionType.None)
			{
				antiGlideMode = false;
			}
		}


		/**
		 * <summary>Gets the character's Animation component.</summary>
		 * <returns>The character's Animation component</returns>
		 */
		public Animation GetAnimation ()
		{
			if (_animation == null)
			{
				_animation = GetComponent<Animation> ();
			}
			return _animation;
		}


		/**
		 * <summary>Gets the character's Animator component. An Animator placed on the spriteChild GameObject will be given priority.</summary>
		 * <returns>The character's Animator component</returns>
		 */
		public Animator GetAnimator ()
		{
			if (_animator == null)
			{
				animatorIsOnRoot = false;

				if (spriteChild && spriteChild.GetComponent<Animator> ())
				{
					_animator = spriteChild.GetComponent<Animator> ();
				}
				else if (customAnimator)
				{
					_animator = customAnimator;
					animatorIsOnRoot = (customAnimator.gameObject == gameObject);
				}
				else
				{
					_animator = GetComponent<Animator> ();
					if (_animator)
					{
						animatorIsOnRoot = true;
					}
				}
			}
			return _animator;
		}


		/**
		 * Resets the internal record of the character's Animator component, which will be re-assigned the next time GetAnimator() is called.
		 */
		public void ResetAnimator ()
		{
			_animator = null;
		}


		private RootMotionType GetRootMotionType ()
		{
			if (charState == CharState.Decelerate)
			{
				//	return RootMotionType.None;
			}

			if (_animator == null || !_animator.applyRootMotion || _animator.runtimeAnimatorController == null || !_animator.enabled)
			{
				return RootMotionType.None;
			}
			if (animEngine.isSpriteBased)
			{
				return RootMotionType.TwoD;
			}

			#if UNITY_EDITOR
			if (_animator && _animator.applyRootMotion && IsPlayer && KickStarter.settingsManager.movementMethod == MovementMethod.FirstPerson && KickStarter.stateHandler.IsInGameplay ())
			{
				ACDebug.LogWarning ("Attempting to movea first-person Player in Root Motion - uncheck 'Apply Root Motion' in the Animator to unconstrain movement.", this);
			}
			#endif

			return RootMotionType.ThreeD;
		}


		/**
		 * <summary>Gets the direction that the character is moving in.</summary>
		 * <returns>The direction that the character is moving in</summary>
		 */
		public Vector3 GetMoveDirection ()
		{
			return moveDirection;
		}


		private void SetNextNodes (bool justResumedPath = false)
		{
			lastDist = Mathf.Infinity;
			pausePath = false;
			nodeActionList = null;

			int tempPrev = targetNode;

			if (IsPlayer && KickStarter.stateHandler.IsInGameplay ())
			{
				targetNode = activePath.GetNextNode (targetNode, prevNode, true);
			}
			else
			{
				targetNode = activePath.GetNextNode (targetNode, prevNode, false);
			}

			prevNode = tempPrev;

			if (targetNode == 0 && activePath.pathType == AC_PathType.Loop && activePath.teleportToStart)
			{
				Teleport (activePath.Transform.position);

				// Set rotation if there is more than one node
				if (activePath.nodes.Count > 1)
				{
					SetLookDirection (activePath.nodes[1] - activePath.nodes[0], true);
				}
				SetNextNodes ();
				return;
			}

			if (targetNode == -1)
			{
				EndPath ();
				return;
			}

			if (justResumedPath && turnBeforeWalking)
			{
				TurnBeforeWalking ();
			}
		}


		/**
		 * <summary>Checks if the character is pathfinding, and the next node on the path will be their intended destination</summary>
		 * <returns>True if the character is pathfinding, and the next node on the path will be their intended destination</returns>
		 */
		public bool WillStopAtNextNode ()
		{
			if (activePath && activePath.WillStopAtNextNode (targetNode))
			{
				return true;
			}
			return false;
		}


		/**
		 * <summary>Checks if the character is currently pathfinding to a pre-determined destination</summary>
		 * <returnsy>True if the character is currently pathfinding to a pre-determined destination</returns>
		 */
		public bool IsPathfinding ()
		{
			if (activePath && activePath == ownPath)
			{
				return true;
			}
			return false;
		}


		/**
		 * <summary>Stops the character from moving along the current Paths object.</summary>
		 * <param name = "optionalPath">If set, the character will only stop if moving along this path</param>
		 */
		public void EndPath (Paths optionalPath, bool stopTurningToo = true)
		{
			if (optionalPath && activePath && activePath != optionalPath)
			{
				return;
			}

			if (stopTurningToo)
			{
				StopTurning ();
			}

			if (activePath)
			{
				KickStarter.eventManager.Call_OnCharacterEndPath (this, activePath);
			}

			if (IsPathfinding ())
			{
				activePath.nodes.Clear ();
			}
			else
			{
				lastPathPrevNode = prevNode;
				lastPathTargetNode = targetNode;
				lastPathActivePath = activePath;
			}

			activePath = null;
			targetNode = 0;
			pathfindUpdateTime = 0f;

			if (charState == CharState.Move)
			{
				if (retroPathfinding)
				{
					Halt ();
				}
				else
				{
					charState = CharState.Decelerate;
				}
			}
		}


		/**
		 * Causes the character to slow down and stop moving.
		 */
		public void StartDecelerating ()
		{
			if (retroPathfinding)
			{
				//Halt (false); // Replace with below

				moveSpeed = 0f;
				exactDestination = defaultExactDestination;

				if (charState == CharState.Move || charState == CharState.Decelerate)
				{
					charState = CharState.Idle;
				}
			}
			else
			{
				charState = CharState.Decelerate;
			}
		}
	

		/**
		 * <summary>Stops the character from moving along the current Paths object.</summary>
		 */
		public virtual void EndPath ()
		{
			EndPath (null);
		}


		/**
		 * Resumes moving along the character's last Paths object, if there was one.
		 */
		public void ResumeLastPath ()
		{
			if (lastPathActivePath)
			{
				SetPath (lastPathActivePath, lastPathTargetNode, lastPathPrevNode);
			}
		}
		
		
		protected void SetLastPath (Paths _lastPathActivePath, int _lastPathTargetNode, int _lastPathPrevNode)
		{
			lastPathActivePath = _lastPathActivePath;
			lastPathTargetNode = _lastPathTargetNode;
			lastPathPrevNode = _lastPathPrevNode;
		}
		
		
		/**
		 * <summary>Stops the character moving.</summary>
		 * <param name = "haltTurning">If True, turning will be halted too</param>
		 */
		public void Halt (bool haltTurning = true)
		{
			if (!IsPathfinding () && activePath)
			{
				lastPathPrevNode = prevNode;
				lastPathTargetNode = targetNode;
				lastPathActivePath = activePath;
			}

			activePath = null;
			targetNode = 0;
			moveSpeed = 0f;

			if (haltTurning)
			{
				StopTurning ();
			}

			exactDestination = defaultExactDestination;

			if (charState == CharState.Move || charState == CharState.Decelerate)
			{
				charState = CharState.Idle;
			}
		}
		
		
		/**
		 * Forces the character into an idle state.
		 */
		public void ForceIdle ()
		{
			charState = CharState.Idle;
		}
		
		
		protected void ReverseDirection ()
		{
			int tempPrev = targetNode;
			targetNode = prevNode;
			prevNode = tempPrev;
		}
		
		
		private void PausePath (float pauseTime)
		{
			StartDecelerating ();
			pausePath = true;
			pausePathTime = Time.time + pauseTime;
			nodeActionList = null;
		}
		
		
		private void PausePath (float pauseTime, Cutscene pauseCutscene)
		{
			StartDecelerating ();
			pausePath = true;
			
			if (pauseTime > 0f)
			{
				pausePathTime = Time.time + pauseTime + 1f;
				StartCoroutine (DelayPathCutscene (pauseTime, pauseCutscene));
			}
			else
			{
				pausePathTime = 0f;
				pauseCutscene.Interact ();
				nodeActionList = pauseCutscene;
			}
		}
		
		
		private IEnumerator DelayPathCutscene (float pauseTime, Cutscene pauseCutscene)
		{
			yield return new WaitForSeconds (pauseTime);
			
			pausePathTime = 0f;
			pauseCutscene.Interact ();
			nodeActionList = pauseCutscene;
		}
		
		
		private void PausePath (float pauseTime, ActionListAsset pauseAsset)
		{
			StartDecelerating ();
			pausePath = true;
			
			if (pauseTime > 0f)
			{
				pausePathTime = Time.time + pauseTime + 1f;
				StartCoroutine (DelayPathActionList (pauseTime, pauseAsset));
			}
			else
			{
				pausePathTime = 0f;
				nodeActionList = AdvGame.RunActionListAsset (pauseAsset);
			}
		}
		
		
		private IEnumerator DelayPathActionList (float pauseTime, ActionListAsset pauseAsset)
		{
			yield return new WaitForSeconds (pauseTime);
			
			pausePathTime = 0f;
			nodeActionList = AdvGame.RunActionListAsset (pauseAsset);
		}
		
		
		private void TurnBeforeWalking ()
		{
			if (retroPathfinding)
			{
				if (charState == CharState.Move || charState == CharState.Decelerate)
				{
					charState = CharState.Idle;
				}
			}

			Vector3 direction = activePath.nodes[1] - Transform.position;

			if (SceneSettings.IsUnity2D ())
			{
				SetLookDirection (new Vector3 (direction.x, 0f, direction.y), false);
			}
			else
			{
				SetLookDirection (new Vector3 (direction.x, 0f, direction.z), false);
			}

			isTurningBeforeWalking = true;
			Turn (false);
		}
		

		/**
		 * <summary>Checks if the character is spot-turning before making their move. isTurningBeforeWalking must be True for this to ever be the case.</summary>
		 * <returns>True if the character is spot-turning before maing their move.</returns>
		 */
		public bool IsTurningBeforeWalking ()
		{
			if (IsTurning () && isTurningBeforeWalking && !Mathf.Approximately (turnSpeed, 0f))
			{
				return true;
			}

			isTurningBeforeWalking = false;
			return false;
		}
		
		
		private bool CanTurnBeforeMoving ()
		{
			if (turnBeforeWalking && IsPathfinding () && targetNode <= 1 && activePath.nodes.Count > 1)
			{
				if (charState == CharState.Move && KickStarter.settingsManager.pathfindUpdateFrequency > 0f)
				{
					// In this case, don't turn as path recalc interferes
					return false;
				}
				return true;
			}
			return false;
		}
		
		
		private void SetPath (Paths pathOb, PathSpeed _speed, int _targetNode, int _prevNode)
		{
			if (pathOb != activePath)
			{
				pausePath = false;
				nodeActionList = null;
			}

			activePath = pathOb;
			targetNode = _targetNode;
			prevNode = _prevNode;

			exactDestination = (pathOb.pathType == AC_PathType.ReverseOnly) ?
								pathOb.Transform.position :
								pathOb.nodes [pathOb.nodes.Count-1];

			if (CanTurnBeforeMoving ())
			{
				TurnBeforeWalking ();
			}
			
			if (pathOb)
			{
				if (_speed == PathSpeed.Run)
				{
					isRunning = true;
				}
				else
				{
					isRunning = false;
				}
			}
			
			if (charState == CharState.Custom)
			{
				charState = CharState.Idle;
			}
			
			pathfindUpdateTime = 0f;

			if (pathOb)
			{
				KickStarter.eventManager.Call_OnCharacterSetPath (this, pathOb);
			}
		}
		
		
		/**
		 * <summary>Assigns the character to a Paths object to move along. This is not the same as pathfinding - this assumes a path has already been defined.</summary>
		 * <param name = "pathOb">The Paths object to move along</param>
		 * <param name = "_speed">The speed to move along the path (Walk, Run)</param>
		 */
		public void SetPath (Paths pathOb, PathSpeed _speed)
		{
			if (pathOb && pathOb.nodes.Count > 0)
			{
				if (pathOb.nodes.Count == 1 && pathOb.nodes[0] == Transform.position)
				{
					return;
				}

				int firstIndex = (pathOb == ownPath) ? 1 : 0;
				SetPath (pathOb, _speed, firstIndex, 0);
			}
			else
			{
				SetPath (pathOb, _speed, 0, 0);
			}
		}
		
		
		/**
		 * <summary>Assigns the character to a Paths object to move along. This is not the same as pathfinding - this assumes a path has already been defined.</summary>
		 * <param name = "pathOb">The Paths object to move along</param>
		 */
		public void SetPath (Paths pathOb)
		{
			if (pathOb)
			{
				if (pathOb.nodes.Count > 0)
				{
					int firstIndex = (pathOb == ownPath) ? 1 : 0;

					if (pathOb.pathType == AC_PathType.ReverseOnly)
					{
						firstIndex = pathOb.nodes.Count - 1;
					}

					SetPath (pathOb, pathOb.pathSpeed, firstIndex, 0);
				}
				else
				{
					SetPath (pathOb, pathOb.pathSpeed, 0, 0);
				}
			}
		}
		
		
		/**
		 * <summary>Assigns the character to a Paths object to move along. This is not the same as pathfinding - this assumes a path has already been defined.</summary>
		 * <param name = "pathOb">The Paths object to move along</param>
		 * <param name = "_targetNode">The index number of the first node to move to</param>
		 * <param name = "_prevNode">The index number of the node moving away from</param>
		 */
		public void SetPath (Paths pathOb, int _targetNode, int _prevNode)
		{
			if (pathOb)
			{
				SetPath (pathOb, pathOb.pathSpeed, _targetNode, _prevNode);
			}
		}
		
		
		/**
		 * <summary>Assigns the character to a Paths object to move along. This is not the same as pathfinding - this assumes a path has already been defined.</summary>
		 * <param name = "pathOb">The Paths object to move along</param>
		 * <param name = "_targetNode">The index number of the first node to move to</param>
		 * <param name = "_prevNode">The index number of the node moving away from</param>
		 * <param name = "affectY">If True, then the character will account for the "Y" position of nodes</param>
		 */
		public void SetPath (Paths pathOb, int _targetNode, int _prevNode, bool affectY)
		{
			if (pathOb)
			{
				SetPath (pathOb, pathOb.pathSpeed, _targetNode, _prevNode);
				activePath.affectY = affectY;
			}
		}

		
		/**
		 * <summary>Gets the current Paths object that the character is moving along. If the character is pathfinding, the Paths component on the character's root will be returned.</summary>
		 * <returns>The current Paths object that the character is moving along.</returns>
		 */
		public Paths GetPath ()
		{
			return activePath;
		}
		

		/** The next node to reach on the character's current path or Paths object */
		public int GetTargetNode ()
		{
			return targetNode;
		}
		

		/** The last-reached node on the character's current path or Paths object */
		public int GetPreviousNode ()
		{
			return prevNode;
		}
		
		
		protected Paths GetLastPath ()
		{
			return lastPathActivePath;
		}
		
		
		protected int GetLastTargetNode ()
		{
			return lastPathTargetNode;
		}
		
		
		protected int GetLastPrevNode ()
		{
			return lastPathPrevNode;
		}
		
		
		/**
		 * <summary>Moves towards a point in the scene.</summary>
		 * <param name = "point">The point to move to</param>
		 * <param name = "run">If True, the character will run</param>
		 * <param name = "usePathfinding">If True, the character will pathfind using the scene's chosen navigation algorithm</param>
		 */
		public void MoveToPoint (Vector3 point, bool run = false, bool usePathfinding = false)
		{
			if (KickStarter.navigationManager == null)
			{
				usePathfinding = false;
			}

			if (usePathfinding)
			{
				Vector3[] pointArray = null;
				pointArray = KickStarter.navigationManager.navigationEngine.GetPointsArray (Transform.position, point, this);
				MoveAlongPoints (pointArray, run);
			}
			else
			{
				List<Vector3> pointData = new List<Vector3>();
				pointData.Add (point);
				MoveAlongPoints (pointData.ToArray (), run, false);
			}
		}


		/**
		 * <summary>Moves along a series of points in the scene.</summary>
		 * <param name = "pointData">The array of points to move along</param>
		 * <param name = "run">If True, the character will run</param>
		 * <param name = "allowUpdating">If True, it will be recalculated over time according to SettingsManager's pathfindUpdateFrequency value</param>
		 */
		public void MoveAlongPoints (Vector3[] pointData, bool run, bool allowUpdating = true)
		{
			if (pointData.Length == 0)
			{
				return;
			}

			Paths path = ownPath;
			if (path)
			{
				path.BuildNavPath (pointData);
				if (run)
				{
					SetPath (path, PathSpeed.Run);
				}
				else
				{
					SetPath (path, PathSpeed.Walk);
				}

				if (allowUpdating)
				{
					pathfindUpdateTime = Mathf.Max (0f, KickStarter.settingsManager.pathfindUpdateFrequency);
				}
				else
				{
					pathfindUpdateTime = -1f;
				}
			}
			else
			{
				ACDebug.LogWarning (this.name + " cannot pathfind without a Paths component", gameObject);
			}
		}
		
		
		/**
		 * Removes all AnimationClips from the Animation component that are not "standard", e.g. Idle, Walk, Run and Talk.
		 */
		public void ResetBaseClips ()
		{
			// Remove all animations except Idle, Walk, Run and Talk
			
			if (spriteChild && spriteChild.GetComponent <Animation>())
			{
				List <string> clipsToRemove = new List <string>();
				
				foreach (AnimationState state in spriteChild.GetComponent <Animation>())
				{
					if ((idleAnim == null || state.name != idleAnim.name) && (walkAnim == null || state.name != walkAnim.name) && (runAnim == null || state.name != runAnim.name))
					{
						clipsToRemove.Add (state.name);
					}
				}
				
				foreach (string _clip in clipsToRemove)
				{
					spriteChild.GetComponent <Animation>().RemoveClip (_clip);
				}
			}
			
			if (_animation)
			{
				List <string> clipsToRemove = new List <string>();
				
				foreach (AnimationState state in _animation)
				{
					if ((idleAnim == null || state.name != idleAnim.name) && (walkAnim == null || state.name != walkAnim.name) && (runAnim == null || state.name != runAnim.name))
					{
						clipsToRemove.Add (state.name);
					}
				}
				
				foreach (string _clip in clipsToRemove)
				{
					_animation.RemoveClip (_clip);
				}
			}
			
		}
		
		
		/**
		 * <summary>Gets the angle that a 2D character's sprite is facing, relative to the root.</summary>
		 * <returns>The angle that a 2D character's sprite is facing, relative to the root</returns>
		 */
		public float GetSpriteAngle ()
		{
			return spriteAngle;
		}


		protected string GetSpriteDirectionToSave ()
		{
			if (lockDirection && flipFrames)
			{
				if (frameFlipping == AC_2DFrameFlipping.LeftMirrorsRight && spriteDirection.Contains ("R"))
				{
					return "L";
				}
				else if (frameFlipping == AC_2DFrameFlipping.RightMirrorsLeft && spriteDirection.Contains ("L"))
				{
					return "R";
				}
			}
			return spriteDirection;
		}


		private string GetSpriteDirectionSuffix (bool ignoreFrameFlipping = false)
		{
			if (ignoreFrameFlipping && flipFrames)
			{
				if (spriteDirection.Contains ("L"))
				{
					return spriteDirection.Replace ("L", "R");
				}
				else if (spriteDirection.Contains ("R"))
				{
					return spriteDirection.Replace ("R", "L");
				}
			}
			return spriteDirection;
		}
		
		
		/**
		 * <summary>Gets the suffix to add to sprite animations that account for the facing direction - e.g. "Walk" -> "Walk_DR".</summary>
		 * <returns>Gets the suffix to add to sprite animations that account for the facing direction - e.g. "Walk" -> "Walk_DR".</returns>
		 */
		public string GetSpriteDirection (bool ignoreFrameFlipping = false)
		{
			return ("_" + GetSpriteDirectionSuffix (ignoreFrameFlipping));
		}
		
		
		/**
		 * <summary>Gets the direction that a 2D character is facing (Down = 0, Left = 1, Right = 2, Up = 3, DownLeft = 4, DownRight = 5, UpLeft = 6, UpRight = 7)</summary>
		 * <returns>The direction that a 2D character is facing (Down = 0, Left = 1, Right = 2, Up = 3, DownLeft = 4, DownRight = 5, UpLeft = 6, UpRight = 7)</returns>
		 */
		public int GetSpriteDirectionInt (bool ignoreFrameFlipping = false)
		{
			switch (GetSpriteDirectionSuffix (ignoreFrameFlipping))
			{
				case "D":
					return 0;

				case "L":
					return 1;

				case "R":
					return 2;

				case "U":
					return 3;

				case "DL":
					return 4;

				case "DR":
					return 5;

				case "UL":
					return 6;

				case "UR":
					return 7;
			}
			return 0;
		}
		
		
		/**
		 * <summary>Locks a 2D character's sprite to face a particular direction.</summary>
		 * <param name = "direction">The direction to face (Down, Left, Right, Up, DownLeft, DownRight, UpLeft, UpRight)</param>
		 */
		public void SetSpriteDirection (CharDirection direction)
		{
			lockDirection = true;

			switch (direction)
			{
				case CharDirection.Down:
					spriteDirection = "D";
					break;

				case CharDirection.Left:
					spriteDirection = "L";
					break;

				case CharDirection.Right:
					spriteDirection = "R";
					break;

				case CharDirection.Up:
					spriteDirection = "U";
					break;

				case CharDirection.DownLeft:
					spriteDirection = "DL";
					break;

				case CharDirection.DownRight:
					spriteDirection = "DR";
					break;

				case CharDirection.UpLeft:
					spriteDirection = "UL";
					break;

				case CharDirection.UpRight:
					spriteDirection = "UR";
					break;

				default:
					break;
			}

			UpdateFrameFlipping (true);
		}
		
		
		private void CalcHeightChange ()
		{
			float currentHeight = Transform.position.y;
			heightChange = currentHeight - prevHeight;
			prevHeight = currentHeight;
		}
		
		
		private void StopStandardAudio ()
		{
			if (audioSource && audioSource.isPlaying)
			{
				if ((runSound && audioSource.clip == runSound) || (walkSound && audioSource.clip == walkSound))
				{
					audioSource.Stop ();
				}
			}
		}
		
		
		private void PlayStandardAudio ()
		{
			if (audioSource)
			{
				if (isRunning && runSound)
				{
					if (audioSource.isPlaying && audioSource.clip == runSound)
					{
						return;
					}

					if (!IsGrounded ())
					{
						return;
					}
					
					audioSource.loop = false;
					audioSource.clip = runSound;
					audioSource.Play ();
					if (KickStarter.eventManager) KickStarter.eventManager.Call_OnPlayFootstepSound (this, null, false, audioSource, runSound);
				}
				else if (walkSound)
				{
					if (audioSource.isPlaying && audioSource.clip == walkSound)
					{
						return;
					}

					if (!IsGrounded ())
					{
						return;
					}
					
					audioSource.loop = false;
					audioSource.clip = walkSound;
					audioSource.Play ();
					if (KickStarter.eventManager) KickStarter.eventManager.Call_OnPlayFootstepSound (this, null, true, audioSource, walkSound);
				}
			}
		}
		
		
		private void ResetAnimationEngine ()
		{
			string className = "AnimEngine";
			
			if (animationEngine == AnimationEngine.Custom)
			{
				if (!string.IsNullOrEmpty (customAnimationClass))
				{
					className = customAnimationClass;
				}
			}
			else
			{
				className += "_" + animationEngine.ToString ();
			}
			
			if (animEngine == null || animEngine.ToString () != className)
			{
				try
				{
					animEngine = (AnimEngine) ScriptableObject.CreateInstance (className);
					if (animEngine != null)
					{
						animEngine.Declare (this);
					}
				}
				catch {}
			}
		}


		private void InitSpriteChild ()
		{
			if (animEngine == null) ResetAnimationEngine ();
			PrepareSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
			UpdateSpriteChild (SceneSettings.IsTopDown (), SceneSettings.IsUnity2D ());
		}

		
		protected void PrepareSpriteChild (bool isTopDown, bool isUnity2D)
		{
			float forwardAmount = 0f;
			float rightAmount = 0f;
			
			if (isTopDown || isUnity2D)
			{
				forwardAmount = Vector3.Dot (Vector3.forward, TransformForward.normalized);
				rightAmount = Vector3.Dot (Vector3.right, TransformForward.normalized);
			}
			else
			{
				forwardAmount = Vector3.Dot (MainCamera.ForwardVector ().normalized, TransformForward.normalized);
				rightAmount = Vector3.Dot (MainCamera.RightVector ().normalized, TransformForward.normalized);
			}
			
			spriteAngle = Mathf.Atan (rightAmount / forwardAmount) * Mathf.Rad2Deg;
			if (forwardAmount > 0f) spriteAngle += 180f;
			else if (rightAmount > 0f) spriteAngle += 360f;

			if (Mathf.Approximately (spriteAngle, 360f))
			{
				spriteAngle = 0f;
			}

			if (charState == CharState.Custom && !flipCustomAnims)
			{
				flipFrames = false;
			}
			else
			{
				if (!lockDirection)
				{
					spriteDirection = spriteDirectionData.GetDirectionalSuffix (spriteAngle);
				}

				UpdateFrameFlipping ();
			}

			if (angleSnapping != AngleSnapping.None)
			{
				spriteAngle = FlattenSpriteAngle (spriteAngle, angleSnapping);
			}
		}


		protected void UpdateFrameFlipping (bool ignoreLockDirectionOption = false)
		{
			if (!ignoreLockDirectionOption && lockDirection) return;

			if (!spriteDirectionData.HasDirections ())
			{
				flipFrames = false;
			}
			else if (frameFlipping == AC_2DFrameFlipping.LeftMirrorsRight && spriteDirection.Contains ("L"))
			{
				spriteDirection = spriteDirection.Replace ("L", "R");
				flipFrames = true;
			}
			else if (frameFlipping == AC_2DFrameFlipping.RightMirrorsLeft && spriteDirection.Contains ("R"))
			{
				spriteDirection = spriteDirection.Replace ("R", "L");
				flipFrames = true;
			}
			else
			{
				flipFrames = false;
			}
		}


		/**
		 * <summary>Snaps a given angle to the nearest 90 or 45 degrees</summary>
		 * <param name = "angle">The angle to snap</param>
		 * <param name = "_angleSnapping">The neareset angle it should be snapped to (FortyFiveDegrees, NinetyDegrees)</param>
		 * <returns>The angle, snapped to the nearest 90 or 45 degrees</returns>
		 */
		public float FlattenSpriteAngle (float angle, AngleSnapping _angleSnapping)
		{
			if (_angleSnapping == AngleSnapping.FortyFiveDegrees)
			{
				if (angle > 337.5f)
				{
					return 360f;
				}
				else if (angle > 292.5f)
				{
					return 315f;
				}
				else if (angle > 247.5f)
				{
					return 270f;
				}
				else if (angle > 202.5f)
				{
					return 225f;
				}
				else if (angle > 157.5f)
				{
					return 180f;
				}
				else if (angle > 112.5f)
				{
					return 135f;
				}
				else if (angle > 67.5f)
				{
					return 90f;
				}
				else if (angle > 22.5f)
				{
					return 45f;
				}
				return 0f;
			}
			else if (_angleSnapping == AngleSnapping.NinetyDegrees)
			{
				if (angle > 315f)
				{
					return 360f;
				}
				else if (angle > 225f)
				{
					return 270f;
				}
				else if (angle > 135f)
				{
					return 180f;
				}
				else if (angle > 45f)
				{
					return 90f;
				}
				return 0f;
			}
			return angle;
		}


		protected void UpdateSpriteChild (bool isTopDown, bool isUnity2D)
		{
			if (frameFlipping != AC_2DFrameFlipping.None)
			{
				if ((flipFrames && spriteChild.localScale.x > 0f) || (!flipFrames && spriteChild.localScale.x < 0f))
				{
					spriteChild.localScale = new Vector3 (-spriteChild.localScale.x, spriteChild.localScale.y, spriteChild.localScale.z);
				}
			}
			
			if (isTopDown)
			{
				if (animEngine && !animEngine.isSpriteBased)
				{
					spriteChild.rotation = TransformRotation;
					spriteChild.RotateAround (Transform.position, Vector3.right, 90f);
				}
				else
				{
					spriteChild.rotation = Quaternion.Euler (90f, 0f, spriteChild.localEulerAngles.z);
				}
			}
			else if (isUnity2D)
			{
				spriteChild.rotation = Quaternion.Euler (0f, 0f, spriteChild.localEulerAngles.z);
			}
			else
			{
				switch (rotateSprite3D)
				{
					case RotateSprite3D.RelativePositionToCamera:
						Vector3 relative = (Transform.position - KickStarter.mainCamera.Transform.position).normalized;
						spriteChild.forward = relative;
						break;

					case RotateSprite3D.CameraFacingDirection:
						spriteChild.rotation = Quaternion.Euler (spriteChild.rotation.eulerAngles.x, KickStarter.mainCamera.Transform.rotation.eulerAngles.y, spriteChild.rotation.eulerAngles.z);
						break;

					case RotateSprite3D.FullCameraRotation:
						spriteChild.rotation = Quaternion.Euler (KickStarter.mainCamera.Transform.rotation.eulerAngles.x, KickStarter.mainCamera.Transform.rotation.eulerAngles.y, KickStarter.mainCamera.Transform.rotation.eulerAngles.z);
						break;

					default:
						break;
				}
			}
		}


		protected void UpdateScale ()
		{
			if (followSortingMap)
			{	
				if (!lockScale)
				{
					spriteScale = followSortingMap.GetLocalScale ();
				}
				
				if (spriteScale > 0f)
				{
					Transform.localScale = originalScale * spriteScale;
				}
					
				if (lockScale)
				{
					sortingMapScale = spriteScale;
				}
				else
				{
					sortingMapScale = followSortingMap.GetLocalSpeed ();
				}
			}
		}
		
		
		/**
		 * <summary>Locks the Renderer's sortingOrder.</summary>
		 * <param name = "order">The sorting order to lock the Renderer to</param>
		 */
		public void SetSorting (int order)
		{
			if (followSortingMap)
			{
				followSortingMap.LockSortingOrder (order);
			}
			else if (GetComponentInChildren <Renderer>())
			{
				GetComponentInChildren <Renderer>().sortingOrder = order;
			}
		}
		
		
		/**
		 * <summary>Locks the Renderer's sorting layer.</summary>
		 * <param name = "layer">The sorting layer to lock the Renderer to</param>
		 */
		public void SetSorting (string layer)
		{
			if (followSortingMap)
			{
				followSortingMap.LockSortingLayer (layer);
			}
			else if (GetComponentInChildren <Renderer>())
			{
				GetComponentInChildren <Renderer>().sortingLayerName = layer;
			}
		}
		
		
		/**
		 * Unlocks any FollowSortingMap that was locked to a sorting order/layer by SetSorting().
		 */
		public void ReleaseSorting ()
		{
			if (followSortingMap)
			{
				followSortingMap.lockSorting = false;
			}
		}
		
		
		/**
		 * <summary>Gets the current movement speed.</summary>
		 * <param name = "reduceNonFacing">If True, then movement speed will be reduced if the character is not facing their indented target.</param>
		 * <returns>The current movement speed</returns>
		 */
		public float GetMoveSpeed (bool reduceNonFacing = false)
		{
			if (isExactLerping && charState == CharState.Decelerate)
			{
				return 0f;
			}

			if (isReversing)
			{
				return -moveSpeed * reverseSpeedFactor;
			}

			if (reduceNonFacing)
			{
				float targetNonFacingFactor = GetNonFacingReductionFactor ();
				nonFacingFactor = nonFacingFactorLerp.Update (nonFacingFactor, targetNonFacingFactor, 10f);
			}
			return moveSpeed * 
					((reduceNonFacing) ? nonFacingFactor : 1f) * 
					((doWallReduction) ? wallReductionFactor : 1f);
		}
		
		
		/**
		 * <summary>Controls the head-facing position.</summary>
		 * <param name = "_headTurnTarget">The Transform to face</param>
		 * <param name = "_headTurnTargetOffset">The position offset of the Transform</param>
		 * <param name = "isInstant">If True, the head will turn instantly</param>
		 * <param name = "_headFacing">What the head should face (Manual, Hotspot, None)</param>
		 */
		public virtual void SetHeadTurnTarget (Transform _headTurnTarget, Vector3 _headTurnTargetOffset, bool isInstant, HeadFacing _headFacing = HeadFacing.Manual)
		{
			if (_headFacing == HeadFacing.Hotspot && headFacing == HeadFacing.Manual)
			{
				// Don't look at Hotspots if manually-set
				return;
			}

			bool isNew = (headTurnTarget != _headTurnTarget || headTurnTargetOffset != _headTurnTargetOffset || headFacing == HeadFacing.None);

			headTurnTarget = _headTurnTarget;
			headTurnTargetOffset = _headTurnTargetOffset;
			headFacing = _headFacing;

			if (isInstant)
			{
				CalculateHeadTurn ();
				SnapHeadMovement ();
			}
			
			if (isNew || isInstant)
			{
				KickStarter.eventManager.Call_OnSetHeadTurnTarget (this, headTurnTarget, headTurnTargetOffset, isInstant);
			}
		}

		
		/**
		 * <summary>Ceases a particular type of head-facing.</summary>
		 * <param name = "_headFacing">The type of head-facing to cease (Hotspot, Manual)</param>
		 * <param name = "isInstant">If True, the head will return to normal instantly</param>
		 */
		public void ClearHeadTurnTarget (bool isInstant, HeadFacing _headFacing)
		{
			if (headFacing == _headFacing)
			{
				ClearHeadTurnTarget (isInstant);
			}
		}
		
		
		/**
		 * <summary>Stops the head from facing a fixed point.</summary>
		 * <param name = "isInstant">If True, the head will return to normal instantly</param>
		 */
		public void ClearHeadTurnTarget (bool isInstant)
		{
			headFacing = HeadFacing.None;
			
			if (isInstant)
			{
				targetHeadAngles = Vector2.zero;
				SnapHeadMovement ();
				headTurnWeight = 0f;
			}

			KickStarter.eventManager.Call_OnClearHeadTurnTarget (this, isInstant);
		}
		
		
		/**
		 * Snaps the head so that it faces the fixed point it's been told to, instead of rotating towards it over time.
		 */
		public void SnapHeadMovement ()
		{
			headTurnWeight = 1f;
			actualHeadAngles = targetHeadAngles;
			AnimateHeadTurn ();
		}
		
		
		/**
		 * <summary>Checks if the head is rotating to face its target, but has not yet reached it.</summary>
		 * <returns>True if the head is rotating to face its target, but has not yet reached it.</returns>
		 */
		public bool IsMovingHead ()
		{
			if (Mathf.Abs (actualHeadAngles.x - targetHeadAngles.x) < 0.1f &&
				Mathf.Abs (actualHeadAngles.y - targetHeadAngles.y) < 0.1f)
			{
				return false;
			}
			return true;
		}
		
		
		/**
		 * <summary>Gets the Shapeable script component attached to the GameObject's root or child.</summary>
		 * <returns>The Shapeable script component attached to the GameObject's root or child.</returns>
		 */
		public Shapeable GetShapeable ()
		{
			Shapeable shapeable = GetComponent <Shapeable> ();
			if (shapeable == null)
			{
				shapeable = GetComponentInChildren <Shapeable>();
			}
			return shapeable;
		}
		
		
		/**
		 * <summary>Gets the position of where the character's head is facing towards.</summary>
		 * <returns>The position of where the character's head is facing towards, or Vector3.zero if their head is not facing anything.</returns>
		 */
		public Vector3 GetHeadTurnTarget ()
		{
			if (headFacing != HeadFacing.None && headTurnTarget)
			{
				return headTurnTarget.position + headTurnTargetOffset;
			}
			return Vector3.zero;
		}
		
		
		private void CalculateHeadTurn ()
		{
			if (headFacing == HeadFacing.None || headTurnTarget == null)
			{
				targetHeadAngles = targetHeadAnglesLerp.Update (targetHeadAngles, Vector2.zero, headTurnSpeed);
				headTurnWeight = headTurnWeightLerp.Update (headTurnWeight, 0f, headTurnSpeed);
			}
			else
			{
				headTurnWeight = headTurnWeightLerp.Update (headTurnWeight, 1f, headTurnSpeed);

				// Horizontal
				Vector3 pointForward = (SceneSettings.IsUnity2D ())
										?
										new Vector3 (headTurnTarget.position.x - Transform.position.x, 0, headTurnTarget.position.y - Transform.position.y) + headTurnTargetOffset
										:
										headTurnTarget.position + headTurnTargetOffset - Transform.position;
				pointForward.y = 0f;
				targetHeadAngles.x = Vector3.Angle (TransformForward, pointForward);
				targetHeadAngles.x = Mathf.Min (targetHeadAngles.x, (animEngine.isSpriteBased) ? 100f : 70f);
				
				Vector3 crossProduct = Vector3.Cross (TransformForward, pointForward);
				float sideOn = Vector3.Dot (crossProduct, Vector2.up);
				
				if (sideOn < 0f)
				{
					targetHeadAngles.x *= -1f;
				}
				
				// Vertical
				Vector3 pointPitch = headTurnTarget.position + headTurnTargetOffset;
				if (neckBone)
				{
					pointPitch -= neckBone.position;
				}
				else
				{
					pointPitch -= Transform.position;
					if (_characterController)
					{
						pointPitch -= new Vector3 (0f, _characterController.height * Transform.localScale.y * 0.8f, 0f);
					}
					else if (capsuleCollider)
					{
						pointPitch -= new Vector3 (0f, capsuleCollider.height * Transform.localScale.y * 0.8f, 0f);
					}
				}

				targetHeadAngles.y = Vector3.Angle (pointPitch, pointForward);
				targetHeadAngles.y = Mathf.Min (targetHeadAngles.y, 50f);

				if (pointPitch.y < pointForward.y)
				{
					targetHeadAngles.y *= -1f;
				}

				if (!ikHeadTurning)
				{
					targetHeadAngles.x /= 60f;

					targetHeadAngles.y *= (Vector3.Dot (TransformForward, pointForward.normalized) / 2f) + 0.5f;
					targetHeadAngles.y /= 30f;
				}
			}
		}
		

		private void AnimateHeadTurn ()
		{
			if (!ikHeadTurning)
			{
				GetAnimEngine ().TurnHead (actualHeadAngles);
			}

			float speedFactor = (Mathf.Approximately (headTurnWeight, 0f) && KickStarter.stateHandler && KickStarter.stateHandler.IsInGameplay ())
								? 0.75f
								: 1.25f;

			actualHeadAngles = actualHeadAnglesLerp.Update (actualHeadAngles, targetHeadAngles, headTurnSpeed * speedFactor);
		}
		
		
		/**
		 * <summary>Parents an object to the character's hand</summary>
		 * <param name = "objectToHold">The object to hold</param>
		 * <param name = "hand">Which hand to parent the object to (Left, Right)</param>
		 * <returns>True if the parenting was sucessful</returns>
		 */
		public bool HoldObject (GameObject objectToHold, Hand hand)
		{
			if (objectToHold == null)
			{
				return false;
			}
			
			Transform handTransform;
			if (hand == Hand.Left)
			{
				handTransform = leftHandBone;
				leftHandHeldObject = objectToHold;
			}
			else
			{
				handTransform = rightHandBone;
				rightHandHeldObject = objectToHold;
			}
			
			if (handTransform)
			{
				objectToHold.transform.parent = handTransform;
				objectToHold.transform.localPosition = Vector3.zero;
				return true;
			}
			
			ACDebug.Log ("Character " + GetName () + " cannot hold object " + objectToHold.name + " - no hand bone found.", gameObject);
			return false;
		}
		
		
		/**
		 * Drops any objects held in the character's hands.
		 */
		public void ReleaseHeldObjects ()
		{
			if (leftHandHeldObject && leftHandHeldObject.transform.IsChildOf (transform))
			{
				leftHandHeldObject.transform.parent = null;
			}
			
			if (rightHandHeldObject && rightHandHeldObject.transform.IsChildOf (transform))
			{
				rightHandHeldObject.transform.parent = null;
			}
		}


		/**
		 * <summary>Gets the position of the top of the character, in world-space.</summary>
		 * <returns>The position of the top of the character, in world-space.</returns>
		 */
		public Vector3 GetSpeechWorldPosition ()
		{
			Vector3 worldPosition = Transform.position;

			if (speechMenuPlacement)
			{
				worldPosition = speechMenuPlacement.position;
			}
			else if (_collider && _collider is CapsuleCollider)
			{
				CapsuleCollider capsuleCollder = (CapsuleCollider) _collider;
				float addedHeight = capsuleCollder.height * Transform.localScale.y;
				
				if (_spriteRenderer)
				{
					addedHeight *= spriteChild.localScale.y;
				}
				
				if (SceneSettings.IsTopDown ())
				{
					worldPosition.z += addedHeight;
				}
				else
				{
					worldPosition.y += addedHeight;
				}
			}
			else if (_characterController)
			{
				float addedHeight = _characterController.height * Transform.localScale.y;

				if (SceneSettings.IsTopDown ())
				{
					worldPosition.z += addedHeight;
				}
				else
				{
					worldPosition.y += addedHeight;
				}
			}
			else
			{
				if (spriteChild)
				{
					if (_spriteRenderer)
					{
						worldPosition.y = _spriteRenderer.bounds.extents.y + _spriteRenderer.bounds.center.y;
					}
					else if (spriteChild.GetComponent <Renderer>())
					{
						worldPosition.y = spriteChild.GetComponent <Renderer>().bounds.extents.y + spriteChild.GetComponent <Renderer>().bounds.center.y;
					}
				}
			}

			return worldPosition;
		}
		
		
		/**
		 * <summary>Gets the position of the top of the character, in screen-space.</summary>
		 * <param name = "keepWithinScreen">If False, and the character is behind the camera, then the returned vector will be shifted far to the right in order to hide it</param>
		 * <returns>The position of the top of the character, in screen-space.</returns>
		 */
		public Vector2 GetSpeechScreenPosition (bool keepWithinScreen = true)
		{
			Vector3 worldPosition = GetSpeechWorldPosition ();
			Vector3 screenPosition = KickStarter.CameraMain.WorldToViewportPoint (worldPosition);

			if (screenPosition.z < 0f)
			{
				if (!keepWithinScreen)
				{
					screenPosition.x = 5f;
				}
				else
				{
					screenPosition.x = 0.5f - screenPosition.x;
				}
			}

			return (new Vector3 (screenPosition.x, 1 - screenPosition.y, screenPosition.z));
		}
		
		
		private bool DoRigidbodyMovement ()
		{
			// Don't use Rigidbody's MovePosition etc if the localScale is being set - Unity bug 
			if (_rigidbody && useRigidbodyForMovement && followSortingMap == null)
			{
				if (retroPathfinding && IsMovingAlongPath ())
				{
					return false;
				}
				return true;
			}
			return false;
		}


		private bool DoRigidbodyMovement2D ()
		{
			if (_rigidbody2D && !antiGlideMode && useRigidbody2DForMovement)
			{
				if (retroPathfinding && IsMovingAlongPath ())
				{
					return false;
				}
				return true;
			}
			return false;
		}

		
		private void OnInitialiseScene ()
		{
			headFacing = HeadFacing.None;
			lockDirection = false;
			lockScale = false;
			isLipSyncing = false;
			lipSyncShapes.Clear ();
			ReleaseSorting ();
		}


		private void OnStartSpeech (Speech speech)
		{ 
			if (speech.GetSpeakingCharacter () == this)
			{
				activeSpeech = speech;
			}
		}


		private void OnStopSpeech (Speech speech)
		{ 
			if (speech == activeSpeech || speech.GetSpeakingCharacter () == this)
			{
				activeSpeech = null;
			}
		}


		private void OnManuallyTurnACOff ()
		{
			EndPath ();
		}


		/**
		 * <summary>Begins a lip-syncing animation based on a series of LipSyncShapes.</summary>
		 * <param name = "_lipSyncShapes">The LipSyncShapes to use as the basis for the animation</param>
		 */
		public void StartLipSync (List<LipSyncShape> _lipSyncShapes)
		{
			#if SalsaIsPresent
			if (KickStarter.speechManager.lipSyncMode == LipSyncMode.Salsa2D)
			{
				salsa2D = GetComponent <Salsa2D>();
				if (salsa2D == null)
				{
					ACDebug.LogWarning ("To perform Salsa 2D lipsyncing, Character GameObjects must have the 'Salsa2D' component attached.");
				}
			}
			else
			{
				salsa2D = null;
			}
			#endif
			
			lipSyncShapes = _lipSyncShapes;
			isLipSyncing = true;
		}
		
		
		/**
		 * <summary>Gets the current lip-sync frame.</summary>
		 * <returns>The current lip-sync frame</returns>
		 */
		public int GetLipSyncFrame ()
		{
			#if SalsaIsPresent
			if (isTalking && salsa2D != null)
			{
				return salsa2D.sayIndex;
			}
			return 0;
			#else
			if (isTalking && lipSyncShapes.Count > 0)
			{
				return lipSyncShapes[0].frame;
			}
			return 0;
			#endif
		}
		
		
		/**
		 * <summary>Gets the current lip-sync frame, as a fraction of the total number of phoneme frames.</summary>
		 * <returns>The current lip-sync frame, as a fraction of the total number of phoneme frames</returns>
		 */
		public float GetLipSyncNormalised ()
		{
			#if SalsaIsPresent
			if (salsa2D != null)
			{
				return ((float) salsa2D.sayIndex / (float) (KickStarter.speechManager.phonemes.Count - 1));
			}
			#endif
			
			if (lipSyncShapes.Count > 0)
			{
				return ((float) lipSyncShapes[0].frame / (float) (KickStarter.speechManager.phonemes.Count - 1));
			}
			return 0f;
		}
		
		
		/**
		 * <summary>Checks if the character is playing a lip-sync animation that affects the GameObject.</summary>
		 * <returns>True if the character is playing a lip-sync animation that affects the GameObject</returns>
		 */
		public bool LipSyncGameObject ()
		{
			if (isLipSyncing && KickStarter.speechManager.lipSyncOutput == LipSyncOutput.PortraitAndGameObject)
			{
				return true;
			}
			return false;
		}
		
		
		private void ProcessLipSync ()
		{
			if (lipSyncShapes.Count > 0)
			{
				#if SalsaIsPresent
				if (salsa2D != null)
				{
					if (KickStarter.speechManager.lipSyncOutput == LipSyncOutput.GameObjectTexture && lipSyncTexture)
					{
						lipSyncTexture.SetFrame (GetLipSyncFrame ());
					}
					return;
				}
				#endif
				
				if (Time.time > lipSyncShapes[0].timeIndex)
				{
					if (KickStarter.speechManager.lipSyncOutput == LipSyncOutput.PortraitAndGameObject && shapeable)
					{
						if (lipSyncShapes.Count > 1)
						{
							float moveTime = lipSyncShapes[1].timeIndex - lipSyncShapes[0].timeIndex;
							shapeable.SetActiveKey (lipSyncGroupID, lipSyncShapes[1].frame, 100f, moveTime * lipSyncBlendShapeSpeedFactor, MoveMethod.Smooth, null);
						}
						else
						{
							shapeable.SetActiveKey (lipSyncGroupID, 0, 100f, 0.2f, MoveMethod.Smooth, null);
						}
					}
					else if (KickStarter.speechManager.lipSyncOutput == LipSyncOutput.GameObjectTexture && lipSyncTexture)
					{
						lipSyncTexture.SetFrame (lipSyncShapes[0].frame);
					}

					lipSyncShapes.RemoveAt (0);
				}
			}
		}
		
		
		/** Stops the character speaking.  This is for animation and audio only - to stop the Speech class from running, call the Dialog script's EndSpeechByCharacter method. */
		public void StopSpeaking ()
		{
			if (_animation && !isLipSyncing)
			{
				foreach (AnimationState state in _animation)
				{
					if (state.layer == (int) AnimLayer.Mouth)
					{
						state.normalizedTime = 1f;
						state.weight = 0f;
					}
				}
			}
			
			if (shapeable && KickStarter.speechManager.lipSyncOutput == LipSyncOutput.PortraitAndGameObject)
			{
				shapeable.DisableAllKeys (lipSyncGroupID, 0.1f, MoveMethod.Curved, null);
			}
			else if (lipSyncTexture && KickStarter.speechManager.lipSyncOutput == LipSyncOutput.GameObjectTexture)
			{
				lipSyncTexture.SetFrame (0);
			}

			if (KickStarter.speechManager.lipSyncMode == LipSyncMode.RogoLipSync)
			{
				RogoLipSyncIntegration.Stop (this);
			}
			else if (KickStarter.speechManager.lipSyncMode == LipSyncMode.FaceFX)
			{
				FaceFXIntegration.Stop (this);
			}
			
			lipSyncShapes.Clear ();
			isLipSyncing = false;
			
			if (speechAudioSource)
			{
				speechAudioSource.Stop ();
			}
		}
		
		
		/**
		 * <summary>Gets the ID number of the Expression that has a specific label.</summary>
		 * <param name = "expressionLabel">The label of the Expression to return the ID number of.</param>
		 * <returns>The ID number of the Expression if found, or -1 if not found</returns>
		 */
		public int GetExpressionID (string expressionLabel)
		{
			if (expressionLabel == "None" || expressionLabel == "none") return 99;

			foreach (Expression expression in expressions)
			{
				if (expression.label == expressionLabel)
				{
					return expression.ID;
				}
			}
			return -1;
		}
		
		
		/**
		 * <summary>Gets the ID number of the currently-active Expression.</summary>
		 * <returns>The ID number of the currently-active Expression</returns>
		 */
		public int GetExpressionID ()
		{
			if (currentExpression != null)
			{
				return currentExpression.ID;
			}
			return 0;
		}
		
		
		/**
		 * <summary>Clears the current Expression so that the default portrait icon is used.</summary>
		 */
		public void ClearExpression ()
		{
			if (currentExpression != null)
			{
				currentExpression = null;
				if (animEngine) animEngine.OnSetExpression ();
			}
				
			if (portraitIcon != null)
			{
				portraitIcon.Reset ();
			}

			for (int i=0; i<expressions.Count; i++)
			{
				if (expressions[i].portraitIcon != null)
				{
					expressions[i].portraitIcon.Reset ();
				}
			}
		}


		/** The character's current Expression */
		public Expression CurrentExpression
		{
			get
			{
				return currentExpression;
			}
		}
		
		
		/**
		 * <summary>Changes the active Expression.</summary>
		 * <param name = "ID">The ID number of the Expression, in expressions, to make active.</param>
		 */
		public void SetExpression (int ID)
		{
			currentExpression = null;

			if (ID == 99)
			{
				if (animEngine) animEngine.OnSetExpression ();
				return;
			}

			foreach (Expression expression in expressions)
			{
				if (expression.ID == ID)
				{
					if (currentExpression != expression)
					{
						currentExpression = expression;
						if (animEngine) animEngine.OnSetExpression ();
					}
					return;
				}
			}
			ACDebug.LogWarning ("Cannot find expression with ID = " + ID.ToString () + " on character " + gameObject.name, gameObject);
		}


		/**
		 * <summary>Changes the active Expression.</summary>
		 * <param name = "_name">The name of the Expression, in expressions, to make active.</param>
		 */
		public void SetExpression (string _name)
		{
			foreach (Expression expression in expressions)
			{
				if (expression.label == _name)
				{
					SetExpression (expression.ID);
					return;
				}
			}
			ACDebug.LogWarning ("Cannot find expression with name = " + _name + " on character " + gameObject.name, gameObject);
		}


		/**
		 * <summary>Gets the active portrait that's displayed in a MenuGraphic element when the character speaks.
		 * The portrait can change if an Expression has been defined.</summary>
		 */
		public CursorIconBase GetPortrait ()
		{
			if (useExpressions && currentExpression != null && currentExpression.portraitIcon.texture)
			{
				return currentExpression.portraitIcon;
			}
			return portraitIcon;
		}



		/**
		 * <summary>Checks if the character (if 3D) is in contact with the ground.</summary>
		 * <param name = "reportError">If True, then an error message will be displayed if no means of detecting the ground can be found</param>
		 * <returns>True if the character (if 3D) is in contact with the ground</returns>
		 */
		public bool IsGrounded (bool reportError = false)
		{
			if (_characterController)
			{
				return _characterController.isGrounded;
			}

			if (_collider && _collider.enabled)
			{
				if (capsuleCollider && capsuleCollider.direction != 1)
				{
					return Physics.CheckCapsule (Transform.position + new Vector3 (0f, _collider.bounds.size.x / 2f, 0f),
												 Transform.position + new Vector3 (0f, _collider.bounds.size.x / 4f, 0f),
												 _collider.bounds.size.y * 0.45f, /* was 0.5 */
												 groundCheckLayerMask);
				}

				return Physics.CheckCapsule (Transform.position + new Vector3 (0f, _collider.bounds.size.y - (_collider.bounds.size.x / 2f), 0f),
											 Transform.position + new Vector3 (0f, _collider.bounds.size.x / 4f, 0f),
											 _collider.bounds.size.x * 0.45f, /* was 0.5 */
											 groundCheckLayerMask);
			}

			if (_rigidbody && Mathf.Abs (_rigidbody.velocity.y) > 0.1f)
			{
				return false;
			}

			if (spriteChild && GetAnimEngine ().isSpriteBased)
			{
				return !isJumping;
			}

			if (KickStarter.settingsManager.movementMethod == MovementMethod.FirstPerson && capsuleColliders != null)
			{
				foreach (CapsuleCollider capsuleCollider in capsuleColliders)
				{
					if (!capsuleCollider.isTrigger && capsuleCollider.enabled)
					{
						return Physics.CheckCapsule (Transform.position + new Vector3 (0f, _collider.bounds.size.y - (_collider.bounds.size.x / 2f), 0f), Transform.position + new Vector3 (0f, _collider.bounds.size.x / 4f, 0f), _collider.bounds.size.x / 2f, groundCheckLayerMask);
					}
				}
			}

			if (reportError)
			{
				ACDebug.LogWarning ("Cannot determine if character '" + name + "' is grounded - consider adding a Capsule Collider component to them.", this);
			}
			
			return false;
		}
		

		protected virtual bool CanBeDirectControlled ()
		{
			return false;
		}
		
		
		/**
		 * <summary>Gets the extent to which motion is controlled by the character script (Automatic, JustTurning, Manual).</summary>
		 * <returns>The extent to which motion is controlled by the character script (Automatic, JustTurning, Manual).</returns>
		 */
		public MotionControl GetMotionControl ()
		{
			return motionControl;
		}
		
		
		/**
		 * <summary>Gets the character's destination when walking along a path.</summary>
		 * <param name = "wantFinalDestination">If True, the paths' end-point will be returned; otherwise, the current node target position will be returned.</param>
		 * <returns>The character's destination when walking along a path. If not moving along a path, the character's position will be returned.</returns>
		 */
		public Vector3 GetTargetPosition (bool wantFinalDestination = false)
		{
			if (activePath)
			{
				if (wantFinalDestination)
				{
					if (activePath.nodes.Count > 0)
					{
						return activePath.nodes [activePath.nodes.Count-1];
					}
				}
				else
				{
					if (targetNode >= 0 && activePath.nodes.Count > targetNode)
					{
						return activePath.nodes[targetNode];
					}
				}
			}
			return Transform.position;
		}


		/**
		 * <summary>Gets the character's target rotation.</summary>
		 * <returns>The character's target rotation</returns>
		 */
		public Quaternion GetTargetRotation ()
		{
			return Quaternion.LookRotation (lookDirection, Vector3.up);
		}


		/**
		 * <summary>Gets the character's calculated rotation for this frame.  This is not the same as GetTargetRotation, which is the 'final' rotation, but instead the rotation that a smoothly-rotated character should take.</summary>
		 * <returns>The character's calculated rotation</returns>
		 */
		public Quaternion GetFrameRotation ()
		{
			return newRotation;
		}
		
		
		/**
		 * <summary>Gets the character's AnimEngine ScriptableObject.</summary>
		 * <returns>The character's AnimEngine ScriptableObject</returns>
		 */
		public AnimEngine GetAnimEngine ()
		{
			#if !UNITY_EDITOR
			if (animEngine == null)
			#endif
			{
				ResetAnimationEngine ();
			}
			return animEngine;
		}


		/**
		 * <summary>Sets the character's AnimEngine ScriptableObject.</summary>
		 * <param name = "_animationEngine">The new AnimationEngine (Legacy, Mecanim, SpritesUnity, SpritesUnityComplex, Sprites2DToolkit)</param>
		 * <param name = "customClassName">The name of the ScriptableObject class to rely on for animation, if _animationEngine = AnimationEngine.Custom</param>
		 */
		public void SetAnimEngine (AnimationEngine _animationEngine, string customClassName = "")
		{
			animationEngine = _animationEngine;
			if (animationEngine == AnimationEngine.Custom)
			{
				customAnimationClass = customClassName;
			}
			ResetAnimationEngine ();
		}
		
		
		/**
		 * <summary>Gets the character's change in height over the last frame.</summary>
		 * <returns>The character's change in height over the last frame</summary>
		 */
		public float GetHeightChange ()
		{
			return heightChange;
		}
		
		
		/**
		 * <summary>Checks if the character is moving backwards.</summary>
		 * <returns>True if the character is moving backwards</returns>
		 */
		public bool IsReversing ()
		{
			return isReversing;
		}
		
		
		/**
		 * <summary>Gets the rate at which the character is turning (negative values used when turning anti-clockwise).</summary>
		 * <returns>The rate at which the character is turning (negative values used when turninng anti-clockwise)</returns>
		 */
		public virtual float GetTurnFloat ()
		{
			return turnFloat;
		}


		/**
		 * <summary>Checks if the character is turning.</summary>
		 * <returns>True if the character is turning</returns>
		 */
		public bool IsTurning ()
		{
			if (lookDirection == Vector3.zero || Quaternion.Angle (Quaternion.LookRotation (lookDirection), TransformRotation) < turningAngleThreshold)
			{
				return false;
			}
			return true;
		}
		
		
		/**
		 * <summary>Checks if the character is moving along a path.</summary>
		 * <returns>True if the character is moving along a path</returns>
		 */
		public bool IsMovingAlongPath ()
		{
			if (activePath != null && !pausePath)
			{
				return true;
			}
			return false;
		}


		/**
		 * <summary>Gets the Character's current display name.</summary>
		 * <param name = "languageNumber">The index number of the game's current language</param>
		 * <returns>The Character's current display name</returns>
		 */
		public string GetName (int languageNumber = 0)
		{
			string newName = gameObject.name;
			if (!string.IsNullOrEmpty (speechLabel))
			{
				newName = speechLabel;

				if (languageNumber > 0)
				{
					return KickStarter.runtimeLanguages.GetTranslation (newName, displayLineID, languageNumber, GetTranslationType (0));
				}
			}
			
			return newName;
		}


		/**
		 * <summary>Renames the Character mid-game.</summary>
		 * <param name = "newName">The new name of the character</param>
		 * <param name = "_lineID">The translation ID number assocated with the new name, as set by SpeechManager</param>
		 */
		public void SetName (string newName, int _lineID)
		{
			if (!string.IsNullOrEmpty (newName))
			{
				speechLabel = newName;

				if (_lineID >= 0)
				{
					displayLineID = _lineID;
				}
				else
				{
					displayLineID = lineID;
				}
			}
			else
			{
				Debug.Log ("Character " + GetName () + " cannot have a blank name.", this);
			}
		}


		/**
		 * <summary>Updates the volume level of the Character's speech audio.</summary>
		 * <param name = "volume">The new volume level</param>
		 */
		public void SetSpeechVolume (float volume)
		{
			if (speechSound)
			{
				speechSound.SetVolume (volume);
			}
			else if (speechAudioSource)
			{
				if (KickStarter.settingsManager.volumeControl == VolumeControl.AudioMixerGroups)
				{
					return;
				}
				speechAudioSource.volume = volume;
			}
		}


		/**
		 * <summary>Gets the character's current speech line</summary>
		 * <returns>The character's current speech line</returns>
		 */
		public Speech GetCurrentSpeech ()
		{
			return activeSpeech;
		}


		private void OnApplicationQuit ()
		{
			if (portraitIcon != null)
			{
				portraitIcon.ClearCache ();
			}

			foreach (Expression expression in expressions)
			{
				if (expression.portraitIcon != null)
				{
					expression.portraitIcon.ClearCache ();
				}
			}
		}


		private bool CanPhysicallyRotate
		{
			get
			{
				AnimEngine _animEngine = GetAnimEngine ();
				if (_animEngine != null && _animEngine.isSpriteBased && !turn2DCharactersIn3DSpace && motionControl != MotionControl.Manual)
				{
					return false;
				}
				return true;
			}
		}


		/** The character's rotation.  This is normally the transform.rotation, but if the character is sprite-based and turn2DCharactersIn3DSpace = False, then this is a 'dummy' rotation instead. */
		public Quaternion TransformRotation
		{
			get
			{
				if (CanPhysicallyRotate)
				{
					return Transform.rotation;
				}
				return actualRotation;
			}
			set
			{
				if (CanPhysicallyRotate)
				{
					Transform.rotation = value;
				}

				actualRotation = value;
				actualForward = actualRotation * Vector3.forward;
				actualRight = actualRotation * Vector3.right;
			}
		}


		/** The character's forward vector.  This is normally the transform.forward, but if the character is sprite-based and turn2DCharactersIn3DSpace = False, then this is a 'dummy' forward instead. */
		public Vector3 TransformForward
		{
			get
			{
				if (CanPhysicallyRotate)
				{
					return Transform.forward;
				}
				return actualForward;
			}
		}


		/** The character's right vector.  This is normally the transform.right, but if the character is sprite-based and turn2DCharactersIn3DSpace = False, then this is a 'dummy' forward instead. */
		public Vector3 TransformRight
		{
			get
			{
				if (CanPhysicallyRotate)
				{
					return Transform.right;
				}
				return actualRight;
			}
		}


		/** If True, then the character is mid-jump */
		public bool IsJumping
		{
			get
			{
				return isJumping;
			}
		}


		/**
		 * <summary>Called whenever the character is bound to a Timeline track that's starting, and used to deactivate certain control components to allow them freedom.</summary>
		 * <param name = "director">The PlayableDirector that is playing the Timeline</param>
		 * <param name = "trackIndex">The index number of the track within the director's TimelineAsset that the character appears on</param>
		 */
		public void OnEnterTimeline (PlayableDirector director, int trackIndex)
		{
			if (!isUnderTimelineControl)
			{
				isUnderTimelineControl = true;
				KickStarter.eventManager.Call_OnCharacterTimeline (this, director, trackIndex, true);
			}
		}


		/**
		 * <summary>Called whenever the character is bound to a Timeline track that's ending, and used to re-activate certain control components.</summary>
		 * <param name = "director">The PlayableDirector that is playing the Timeline</param>
		 * <param name = "trackIndex">The index number of the track within the director's TimelineAsset that the character appears on</param>
		 */
		public void OnExitTimeline (PlayableDirector director, int trackIndex)
		{
			if (isUnderTimelineControl)
			{
				isUnderTimelineControl = false;
				KickStarter.eventManager.Call_OnCharacterTimeline (this, director, trackIndex, false);
			}
		}


		/**
		 * <summary>Gets a Sprite based on the portrait graphic of the character.
		 * If lipsincing is enabled, the sprite will be based on the current phoneme.</summary>
		 * <returns>The character's portrait sprite</returns>
		 */
		public Sprite GetPortraitSprite ()
		{
			CursorIconBase portraitIcon = GetPortrait ();
			if (portraitIcon != null && portraitIcon.texture)
			{
				if (portraitIcon.isAnimated)
				{
					if (isLipSyncing)
					{
						return portraitIcon.GetAnimatedSprite (GetLipSyncFrame ());
					}
					else
					{
						return portraitIcon.GetAnimatedSprite (isTalking);
					}
				}
				else
				{
					return portraitIcon.GetSprite ();
				}
			}
			return null;
		}


		#region ITranslatable

		public string GetTranslatableString (int index)
		{
			return speechLabel;
		}


		public int GetTranslationID (int index)
		{
			return lineID;
		}

		
		public AC_TextType GetTranslationType (int index)
		{
			return AC_TextType.Character;
		}


		#if UNITY_EDITOR

		public void UpdateTranslatableString (int index, string updatedText)
		{
			speechLabel = updatedText;
		}


		public int GetNumTranslatables ()
		{
			return 1;
		}


		public bool HasExistingTranslation (int index)
		{
			return (lineID > -1);
		}


		public void SetTranslationID (int index, int _lineID)
		{
			lineID = _lineID;
		}


		public string GetOwner (int index)
		{
			return string.Empty;
		}


		public bool OwnerIsPlayer (int index)
		{
			return false;
		}


		public bool CanTranslate (int index)
		{
			return (!string.IsNullOrEmpty (speechLabel));
		}

		#endif

		#endregion


		/** Data related to what directions the character can face, if sprite-based */
		public SpriteDirectionData spriteDirectionData
		{
			get
			{
				if (_spriteDirectionData == null || !_spriteDirectionData.IsUpgraded)
				{
					_spriteDirectionData = new SpriteDirectionData (doDirections, doDiagonals);

					#if UNITY_EDITOR
					if (GetAnimEngine () != null && GetAnimEngine ().isSpriteBased)
					{
						if (Application.isPlaying)
						{
							ACDebug.Log ("The character '" + gameObject.name + "' has been temporarily upgraded - please exit Play mode and view their Inspector to upgrade permanently.", this);
						}
						else
						{
							UnityVersionHandler.CustomSetDirty (this, true);
						}
					}
					#endif
				}
				return _spriteDirectionData;
			}
		}


		/** The CharacterAnimation2DShot that's currently being applied to the character, if sprite-based */
		public CharacterAnimation2DShot ActiveCharacterAnimation2DShot
		{
			get
			{
				return activeCharacterAnimation2DShot;
			}
			set
			{
				activeCharacterAnimation2DShot = value;
			}
		}


		private bool AnimationControlledByAnimationShot
		{
			get
			{
				return (activeCharacterAnimation2DShot != null);
			}
		}


		public IKLimbController LeftHandIKController
		{
			get
			{
				return leftHandIKController;
			}
		}


		public IKLimbController RightHandIKController
		{
			get
			{
				return rightHandIKController;
			}
		}


		public void SetTimelineHeadTurnOverride (Transform _timelineHeadTurnTarget, Vector3 _timelineHeadTurnTargetOffset, float _timelineHeadTurnWeight)
		{
			if (_timelineHeadTurnTarget == null)
			{
				ReleaseTimelineHeadTurnOverride ();
				return;
			}

			bool isNew = (!timelineHeadTurnOverride || _timelineHeadTurnTarget != timelineHeadTurnTarget);
			
			timelineHeadTurnOverride = true;
			timelineHeadTurnTarget = _timelineHeadTurnTarget;
			timelineHeadTurnTargetOffset = _timelineHeadTurnTargetOffset;
			timelineHeadTurnWeight = _timelineHeadTurnWeight;

			if (isNew)
			{
				KickStarter.eventManager.Call_OnSetHeadTurnTarget (this, _timelineHeadTurnTarget, _timelineHeadTurnTargetOffset, true);
			}

			if (Application.isPlaying && (animEngine == null || !animEngine.IKEnabled))
			{
				ACDebug.LogWarning ("Character " + GetName () + " will not be able to respond to the head-turning Timeline track because they do not have IK head-turning enabled.", this);
			}
		}


		public void ReleaseTimelineHeadTurnOverride ()
		{
			if (timelineHeadTurnOverride)
			{
				timelineHeadTurnOverride = false;
				KickStarter.eventManager.Call_OnClearHeadTurnTarget (this, true);
			}
		}


		/** A cache of the character's root transform component */
		public Transform Transform
		{
			get
			{
				if (_transform == null) _transform = transform;
				return _transform;
			}
		}


		public bool isTalking
		{
			get
			{
				return (activeSpeech != null && activeSpeech.IsAnimating ());
			}
		}

	}

}