/*
 *
 *	Adventure Creator
 *	by Chris Burton, 2013-2021
 *	
 *	"ActionRename.cs"
 * 
 *	This action renames Hotspots. A "Remember Name" script needs to be
 *	attached to said hotspot if the renaming is to carry across saved games.
 * 
 */

using UnityEngine;
using System.Collections.Generic;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace AC
{

	[System.Serializable]
	public class ActionRename : Action, ITranslatable
	{
		
		public int constantID = 0;
		public int parameterID = -1;
		public Hotspot hotspot;
		protected Hotspot runtimeHotspot;

		public string newName;
		public int lineID = -1;


		public override ActionCategory Category { get { return ActionCategory.Hotspot; }}
		public override string Title { get { return "Rename"; }}
		public override string Description { get { return "Renames a Hotspot, or an NPC with a Hotspot component."; }}


		public override void AssignValues (List<ActionParameter> parameters)
		{
			runtimeHotspot = AssignFile <Hotspot> (parameters, parameterID, constantID, hotspot);
		}
		
		
		public override float Run ()
		{
			if (runtimeHotspot && !string.IsNullOrEmpty (newName))
			{
				runtimeHotspot.SetName (newName, lineID);
			}
			
			return 0f;
		}
		
		
		#if UNITY_EDITOR
		
		public override void ShowGUI (List<ActionParameter> parameters)
		{
			parameterID = Action.ChooseParameterGUI ("Hotspot to rename:", parameters, parameterID, ParameterType.GameObject);
			if (parameterID >= 0)
			{
				constantID = 0;
				hotspot = null;
			}
			else
			{
				hotspot = (Hotspot) EditorGUILayout.ObjectField ("Hotspot to rename:", hotspot, typeof (Hotspot), true);
				
				constantID = FieldToID <Hotspot> (hotspot, constantID);
				hotspot = IDToField <Hotspot> (hotspot, constantID, false);
			}
			
			newName = EditorGUILayout.TextField ("New label:", newName);
		}


		public override void AssignConstantIDs (bool saveScriptsToo, bool fromAssetFile)
		{
			if (saveScriptsToo)
			{
				AddSaveScript <RememberHotspot> (hotspot);
			}

			AssignConstantID <Hotspot> (hotspot, constantID, parameterID);
		}
		
		
		public override string SetLabel ()
		{
			if (hotspot != null && !string.IsNullOrEmpty (newName))
			{
				return hotspot.name + " to " + newName;
			}
			return string.Empty;
		}


		public override bool ReferencesObjectOrID (GameObject gameObject, int id)
		{
			if (parameterID < 0)
			{
				if (hotspot != null && hotspot.gameObject == gameObject) return true;
				if (constantID == id && id != 0) return true;
			}
			return base.ReferencesObjectOrID (gameObject, id);
		}
		
		#endif


		#region ITranslatable

		public string GetTranslatableString (int index)
		{
			return newName;
		}


		public int GetTranslationID (int index)
		{
			return lineID;
		}


		#if UNITY_EDITOR

		public void UpdateTranslatableString (int index, string updatedText)
		{
			newName = updatedText;
		}


		public int GetNumTranslatables ()
		{
			return 1;
		}


		public bool HasExistingTranslation (int index)
		{
			return (lineID > -1);
		}


		public void SetTranslationID (int index, int _lineID)
		{
			lineID = _lineID;
		}


		public string GetOwner (int index)
		{
			return string.Empty;
		}


		public bool OwnerIsPlayer (int index)
		{
			return false;
		}


		public AC_TextType GetTranslationType (int index)
		{
			return AC_TextType.Hotspot;
		}


		public bool CanTranslate (int index)
		{
			return (!string.IsNullOrEmpty (newName));
		}

		#endif

		#endregion


		/**
		 * <summary>Creates a new instance of the 'Hotspot: Rename' Action</summary>
		 * <param name = "hotspotToRename">The Hotspot to rename</param>
		 * <param name = "newName">If Hotspot's new name</param>
		 * <param name = "translationID">The new name's translation ID number, as generated by the Speech Manager</param>
		 * <returns>The generated Action</returns>
		 */
		public static ActionRename CreateNew (Hotspot hotspotToRename, string newName, int translationID = -1)
		{
			ActionRename newAction = CreateNew<ActionRename> ();
			newAction.hotspot = hotspotToRename;
			newAction.newName = newName;
			newAction.lineID = translationID;
			return newAction;
		}

	}

}