﻿/*
 *
 *	Adventure Creator
 *	by Chris Burton, 2013-2021
 *	
 *	"ActionCharRename.cs"
 * 
 *	This action renames Hotspots. A "Remember NPC" script needs to be
 *	attached to the Character unless it is a Player prefab.
 * 
 */

using UnityEngine;
using System.Collections.Generic;

#if UNITY_EDITOR
using UnityEditor;
#endif

namespace AC
{

	[System.Serializable]
	public class ActionCharRename : Action, ITranslatable
	{
		
		public bool isPlayer;
		public int playerID = -1;

		public int _charID = 0;
		public Char _char;
		protected Char runtimeChar;

		public string newName;
		public int lineID = -1;


		public override ActionCategory Category { get { return ActionCategory.Character; }}
		public override string Title { get { return "Rename"; }}
		public override string Description { get { return "Changes the display name of a Character when subtitles are used."; }}


		public override void AssignValues (List<ActionParameter> parameters)
		{
			if (isPlayer)
			{
				runtimeChar = AssignPlayer (playerID, parameters, -1);
			}
			else
			{
				runtimeChar = AssignFile <Char> (_charID, _char);
			}
		}
		
		
		public override float Run ()
		{
			if (!string.IsNullOrEmpty (newName))
			{
				if (runtimeChar != null)
				{
					runtimeChar.SetName (newName, lineID);
				}
				else if (playerID >= 0 && KickStarter.settingsManager.playerSwitching == PlayerSwitching.Allow && KickStarter.saveSystem.CurrentPlayerID != playerID)
				{
					// Special case: Player is not in the scene, so manually update their PlayerData
					PlayerData playerData = KickStarter.saveSystem.GetPlayerData (playerID);
					if (playerData != null)
					{
						playerData.playerSpeechLabel = newName;
						playerData.playerDisplayLineID = lineID;
					}
				}
			}
			
			return 0f;
		}
		
		
		#if UNITY_EDITOR
		
		public override void ShowGUI (List<ActionParameter> parameters)
		{
			isPlayer = EditorGUILayout.Toggle ("Is Player?", isPlayer);
			if (isPlayer)
			{
				if (KickStarter.settingsManager != null && KickStarter.settingsManager.playerSwitching == PlayerSwitching.Allow)
				{
					playerID = ChoosePlayerGUI (playerID, true);
				}
			}
			else
			{
				_char = (Char) EditorGUILayout.ObjectField ("Character:", _char, typeof (Char), true);
				
				_charID = FieldToID <Char> (_char, _charID);
				_char = IDToField <Char> (_char, _charID, true);
			}
			
			newName = EditorGUILayout.TextField ("New name:", newName);
		}


		public override void AssignConstantIDs (bool saveScriptsToo, bool fromAssetFile)
		{
			if (!isPlayer)
			{
				if (saveScriptsToo)
				{
					if (_char != null && !_char.IsPlayer)
					{
						AddSaveScript <RememberNPC> (_char);
					}
				}

				AssignConstantID <Char> (_char, _charID, 0);
			}
		}

		
		public override string SetLabel ()
		{
			if (_char != null && !string.IsNullOrEmpty (newName))
			{
				return _char.name + " to " + newName;
			}
			return string.Empty;
		}


		public override bool ReferencesObjectOrID (GameObject _gameObject, int id)
		{
			if (!isPlayer)
			{
				if (_char != null && _char.gameObject == _gameObject) return true;
				if (_charID == id) return true;
			}
			if (isPlayer && _gameObject.GetComponent <Player>() != null) return true;
			return base.ReferencesObjectOrID (_gameObject, id);
		}


		public override bool ReferencesPlayer (int _playerID = -1)
		{
			if (!isPlayer) return false;
			if (_playerID < 0) return true;
			if (playerID < 0) return true;
			return (playerID == _playerID);
		}

		#endif


		#region ITranslatable

		public string GetTranslatableString (int index)
		{
			return newName;
		}


		public int GetTranslationID (int index)
		{
			return lineID;
		}


		#if UNITY_EDITOR

		public void UpdateTranslatableString (int index, string updatedText)
		{
			newName = updatedText;
		}


		public int GetNumTranslatables ()
		{
			return 1;
		}


		public bool HasExistingTranslation (int index)
		{
			return (lineID > -1);
		}


		public void SetTranslationID (int index, int _lineID)
		{
			lineID = _lineID;
		}


		public string GetOwner (int index)
		{
			return string.Empty;
		}


		public bool OwnerIsPlayer (int index)
		{
			return isPlayer;
		}


		public AC_TextType GetTranslationType (int index)
		{
			return AC_TextType.Hotspot;
		}


		public bool CanTranslate (int index)
		{
			return (!string.IsNullOrEmpty (newName));
		}

		#endif

		#endregion


		/**
		 * <summary>Creates a new instance of the 'Character: Rename' Action with key variables already set.</summary>
		 * <param name = "characterToRename">The character to rename</param>
		 * <param name = "newName">The character's new name</param>
		 * <param name = "translationIDs">The line's translation ID number, as generated by the Speech Manager</param>
		 * <returns>The generated Action</returns>
		 */
		public static ActionCharRename CreateNew (Char characterToRename, string newName, int translationID = -1)
		{
			ActionCharRename newAction = CreateNew<ActionCharRename> ();
			newAction._char = characterToRename;
			newAction.newName = newName;
			newAction.lineID = translationID;
			return newAction;
		}

	}

}